<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PageSettingResource\Pages;
use App\Models\PageSetting;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Section;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\BooleanColumn;

class PageSettingResource extends Resource
{
    protected static ?string $model = PageSetting::class;

    protected static ?string $navigationIcon = 'heroicon-o-lock-closed';

    protected static ?string $navigationLabel = 'Page Settings';

    protected static ?string $modelLabel = 'Page Setting';

    protected static ?string $pluralModelLabel = 'Page Settings';

    protected static ?string $navigationGroup = 'Content Management';

    protected static ?int $navigationSort = 4;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Page Information')
                    ->schema([
                        TextInput::make('page_name')
                            ->required()
                            ->maxLength(255)
                            ->label('Page Name')
                            ->helperText('Display name for this page'),

                        TextInput::make('page_url')
                            ->required()
                            ->maxLength(255)
                            ->label('Page URL')
                            ->helperText('Full URL path e.g., /pbl-integrated-report/2025')
                            ->unique(ignoreRecord: true)
                            ->prefix('/'),
                    ])
                    ->columns(2),

                Section::make('Password Protection')
                    ->schema([
                        Toggle::make('is_protected')
                            ->label('Enable Password Protection')
                            ->helperText('When enabled, users must enter a password to access this page')
                            ->live()
                            ->afterStateUpdated(function ($state, Forms\Set $set) {
                                if (!$state) {
                                    $set('password', null);
                                }
                            }),

                        TextInput::make('password')
                            ->label('Password')
                            ->password()
                            ->helperText('Leave empty when editing to keep current password unchanged. Required when creating a new protected page.')
                            ->visible(fn (Forms\Get $get) => $get('is_protected'))
                            ->required(fn (Forms\Get $get, $context) => $get('is_protected') && $context === 'create')
                            ->dehydrated(fn ($state) => filled($state))
                            ->minLength(4)
                            ->maxLength(255),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('page_name')
                    ->label('Page Name')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('page_url')
                    ->label('URL')
                    ->searchable()
                    ->copyable()
                    ->copyMessage('URL copied!'),

                BooleanColumn::make('is_protected')
                    ->label('Protected')
                    ->sortable(),

                TextColumn::make('updated_at')
                    ->label('Last Updated')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_protected')
                    ->label('Password Protected'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPageSettings::route('/'),
            'create' => Pages\CreatePageSetting::route('/create'),
            'edit' => Pages\EditPageSetting::route('/{record}/edit'),
        ];
    }
}
