<?php

namespace App\Http\Controllers\Api;

use App\Models\Brand;
use App\Models\Stock;
use GuzzleHttp\Client;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Mail\ContactFormMail;
use App\Services\StockService;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use App\Models\ContactFormSubmission;

class CompanydataController extends Controller
{

    public function brands(Request $request)
    {
        return response()->json(['brands' => Brand::paginate(12)]);
    }

    public function getlatestStockData(StockService $stockService)
    {
        $data = $stockService->getCurrentStockData();

        return response()->json($data);
    }



    public function getStockData()
    {
        try {
            $stocks = Stock::orderBy('date', 'desc')->take(31)->get()->reverse();
            $formattedStocks = $stocks->map(function ($stock) {
                return [
                    'date' => $stock->date,
                    'price' => (float) $stock->closing_price,
                    'semdex' => (float) $stock->semdex ?? 0,
                ];
            })->values();

            return response()->json($formattedStocks);
        } catch (\Exception $e) {
            Log::error('Error in getStockData: ' . $e->getMessage());
            return response()->json([]);
        }
    }

    public function getStockMetrics(StockService $stockService)
    {
        try {
            // Get current stock data from live API
            $currentData = $stockService->getCurrentStockData();
            
            // Get historical data for calculations
            $stocks = Stock::orderBy('date', 'desc')->take(365)->get();
            
            if ($stocks->isEmpty()) {
                return response()->json([
                    'error' => 'No stock data available'
                ], 404);
            }

            $currentPrice = (float) $currentData['price'];
            
            // Get the most recent database record for previous close
            $previousPrice = (float) $stocks->first()->closing_price;
            
            // If current price is invalid or zero, use the most recent database price
            if ($currentPrice <= 0) {
                $currentPrice = $previousPrice;
                Log::warning("Invalid current price from API, using database price: {$currentPrice}");
            }
            
            // Calculate price change
            $priceChange = $currentPrice - $previousPrice;
            $percentageChange = ($priceChange / $previousPrice) * 100;
            

            
            // Calculate 52-week high/low
            $yearlyData = $stocks->take(252); // Approximate trading days in a year
            $yearlyPrices = $yearlyData->pluck('closing_price')->map(function($price) {
                return (float) $price;
            });
            
            $week52High = $yearlyPrices->max();
            $week52Low = $yearlyPrices->min();
            
            // Calculate average volume (simulated)
            $avgVolume = rand(8000000, 12000000);
            $currentVolume = rand(4000000, 8000000);
            
            // Calculate market cap (simulated based on price)
            $marketCap = $currentPrice * 342000000; // Assuming 342M shares
            
            // Calculate yield (simulated)
            $annualDividend = 4.35; // Simulated annual dividend
            $yield = ($annualDividend / $currentPrice) * 100;
            
            return response()->json([
                'currentPrice' => $currentPrice,
                'priceChange' => round($priceChange, 2),
                'percentageChange' => round($percentageChange, 2),
                'previousClose' => $previousPrice,
                'week52High' => round($week52High, 2),
                'week52Low' => round($week52Low, 2),
                'volume' => number_format($currentVolume),
                'avgVolume' => number_format($avgVolume),
                'marketCap' => $this->formatMarketCap($marketCap),
                'yield' => round($yield, 2),
                'lastUpdated' => $currentData['date'] . ' ' . $currentData['time'],
                'currency' => 'MUR',
                'exchange' => 'SEMDEX'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error getting stock metrics: ' . $e->getMessage());
            return response()->json([
                'error' => 'Failed to fetch stock metrics'
            ], 500);
        }
    }

    private function formatMarketCap($marketCap)
    {
        if ($marketCap >= 1000000000000) {
            return round($marketCap / 1000000000000, 1) . 'T';
        } elseif ($marketCap >= 1000000000) {
            return round($marketCap / 1000000000, 1) . 'B';
        } elseif ($marketCap >= 1000000) {
            return round($marketCap / 1000000, 1) . 'M';
        } else {
            return number_format($marketCap);
        }
    }

    public function getStockDataByPeriod(Request $request)
    {
        try {
            $period = $request->input('period', '1M');
            $stocks = Stock::orderBy('date', 'desc');
            
            $startDate = null;
            
            // Filter by period using actual date ranges for current year
            switch ($period) {
                case '1D':
                    // Show yesterday's data (most recent trading day)
                    $startDate = now()->subDay()->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate)->take(1);
                    break;
                case '5D':
                    // Show last 5 days backwards from now
                    $startDate = now()->subDays(5)->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate);
                    break;
                case '1W':
                    // Show last 7 days backwards from now
                    $startDate = now()->subDays(7)->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate);
                    break;
                case '1M':
                    // Show last 30 days backwards from now
                    $startDate = now()->subDays(30)->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate);
                    break;
                case '3M':
                    // Show last 90 days backwards from now
                    $startDate = now()->subDays(90)->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate);
                    break;
                case '6M':
                    // Show last 180 days backwards from now
                    $startDate = now()->subDays(180)->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate);
                    break;
                case 'YTD':
                    // Show from start of year to now
                    $startDate = date('Y-01-01');
                    $stocks = $stocks->where('date', '>=', $startDate);
                    break;
                case '1Y':
                    // Show last 365 days backwards from now
                    $startDate = now()->subDays(365)->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate);
                    break;
                case '2Y':
                    // Show last 730 days backwards from now
                    $startDate = now()->subDays(730)->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate);
                    break;
                default:
                    // Default to last 30 days
                    $startDate = now()->subDays(30)->format('Y-m-d');
                    $stocks = $stocks->where('date', '>=', $startDate);
            }
            
            $stockCollection = $stocks->get();
            
            // If no data found, return empty array
            if ($stockCollection->isEmpty()) {
                return response()->json([]);
            }
            
            $formattedStocks = $stockCollection->reverse()->map(function ($stock) {
                return [
                    'date' => $stock->date,
                    'price' => (float) $stock->closing_price,
                    'semdex' => (float) $stock->semdex ?? 0,
                ];
            })->values();

            return response()->json($formattedStocks);
            
        } catch (\Exception $e) {
            Log::error('Error in getStockDataByPeriod: ' . $e->getMessage());
            return response()->json([]);
        }
    }


    public function getShareholdersData(StockService $stockService)
    {
        try {
            $pageData = $stockService->getShareholdersPageData();
            
            return response()->json([
                'stock_info' => $pageData['stock_info'],
                'financial_metrics' => $pageData['financial_metrics'],
                'share_price_trend' => $pageData['share_price_trend']
            ]);
        } catch (\Exception $e) {
            Log::error('Error getting shareholders data: ' . $e->getMessage());
            return response()->json([
                'error' => 'Failed to fetch shareholders data'
            ], 500);
        }
    }

    public function debugStockData(StockService $stockService)
    {
        try {
            // Clear cache to force fresh data
            Cache::forget('current_stock_data');
            
            // Get fresh stock data
            $currentData = $stockService->getCurrentStockData();
            
            // Get database data
            $dbStocks = Stock::orderBy('date', 'desc')->take(5)->get();
            
            return response()->json([
                'live_data' => $currentData,
                'database_data' => $dbStocks->map(function($stock) {
                    return [
                        'date' => $stock->date,
                        'closing_price' => $stock->closing_price,
                        'semdex' => $stock->semdex
                    ];
                }),
                'total_db_records' => Stock::count(),
                'cache_cleared' => true
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function sendEmail(Request $request)
    {
        // Spam detection
        $spamDetection = new \App\Services\SpamDetectionService();
        
        if ($spamDetection->isSpam($request, 'contact')) {
            $spamDetection->flagSpamIp($request->ip());
            return response()->json([
                'status' => 'error', 
                'message' => 'Your submission has been flagged as spam. Please try again later.'
            ], 400);
        }

        $validatedData = $request->validate([
            'h-recaptcha-response' => 'required',
            "full_name" => "required|string|min:3|max:50",
            "email" => "required|email|string",
            "message" => "required|string|min:10|max:1000",
            "form_start_time" => "required|integer"
        ]);

        $hcaptchaResponse = $request->input('h-recaptcha-response');

        $client = new Client();
        $response = $client->post('https://hcaptcha.com/siteverify', [
            'form_params' => [
                'secret' => config('captcha.secret'),
                'response' => $hcaptchaResponse,
                'remoteip' => $request->ip(),
            ]
        ]);


        $body = json_decode($response->getBody()->getContents());

        if (!$body->success) {
            return response()->json(['status' => 'error', 'message' => 'hCaptcha verification failed.'], 400);
        }


        $uniqueKey = 'USR-' . strtoupper(Str::random(3)) . '-' . rand(1000, 9999) . '-' . strtoupper(Str::random(3));

        $contactDetails = ContactFormSubmission::create([
            'full_name' => $validatedData['full_name'],
            'email' => $validatedData['email'],
            'message' => $validatedData['message'],
            'contact_form_id' => $uniqueKey,
        ]);

        $adminEmail = config('mail.from.address');
        if ($adminEmail) {
            Mail::to($validatedData['email'])->send(new ContactFormMail($contactDetails));
            Mail::to($adminEmail)->send(new ContactFormMail($contactDetails, $adminEmail));

            return response()->json(['status' => 'success', 'message' => 'success', 'data' => $contactDetails->contact_form_id], 200);
        } else {
            return response()->json(['status' => 'failed', 'message' => 'an error occurred'], 500);
        }
    }
}
