<?php

namespace App\Services;

use App\Models\SeoData;
use Illuminate\Support\Facades\Cache;

class SeoService
{
    protected $title;
    protected $description;
    protected $keywords;
    protected $image;
    protected $url;
    protected $type = 'website';
    protected $author;
    protected $robots = 'index, follow';
    protected $canonical;
    protected $openGraph = [];
    protected $twitterCard = [];
    protected $structuredData = [];
    protected $hreflang = [];
    protected $alternateUrls = [];

    public function setTitle($title)
    {
        $this->title = $title;
        return $this;
    }

    public function setDescription($description)
    {
        $this->description = $description;
        return $this;
    }

    public function setKeywords($keywords)
    {
        $this->keywords = $keywords;
        return $this;
    }

    public function setImage($image)
    {
        $this->image = $image;
        return $this;
    }

    public function setUrl($url)
    {
        $this->url = $url;
        return $this;
    }

    public function setType($type)
    {
        $this->type = $type;
        return $this;
    }

    public function setAuthor($author)
    {
        $this->author = $author;
        return $this;
    }

    public function setRobots($robots)
    {
        $this->robots = $robots;
        return $this;
    }

    public function setCanonical($canonical)
    {
        $this->canonical = $canonical;
        return $this;
    }

    public function setOpenGraph($data)
    {
        $this->openGraph = $data;
        return $this;
    }

    public function setTwitterCard($data)
    {
        $this->twitterCard = $data;
        return $this;
    }

    public function setStructuredData($data)
    {
        $this->structuredData = $data;
        return $this;
    }

    /**
     * Load SEO data from database by route with caching
     */
    public function loadFromDatabase($route)
    {
        $cacheKey = "seo_data_{$route}";
        $cacheDuration = 3600; // 1 hour
        
        $seoData = Cache::remember($cacheKey, $cacheDuration, function() use ($route) {
            return SeoData::getByRoute($route);
        });
        
        if ($seoData) {
            $this->setTitle($seoData->title)
                 ->setDescription($seoData->description)
                 ->setKeywords($seoData->keywords)
                 ->setImage($seoData->image)
                 ->setUrl($seoData->url)
                 ->setType($seoData->type)
                 ->setAuthor($seoData->author)
                 ->setRobots($seoData->robots)
                 ->setCanonical($seoData->canonical)
                 ->setOpenGraph($seoData->open_graph ?? [])
                 ->setTwitterCard($seoData->twitter_card ?? [])
                 ->setStructuredData($seoData->structured_data ?? [])
                 ->setHreflang($seoData->hreflang ?? [])
                 ->setAlternateUrls($seoData->alternate_urls ?? []);
        }
        
        return $this;
    }

    /**
     * Create or update SEO data in database
     */
    public function saveToDatabase($route, $data)
    {
        return SeoData::updateOrCreate(
            ['page_route' => $route],
            array_merge($data, ['page_route' => $route])
        );
    }

    public function render()
    {
        $meta = [];

        // Basic meta tags
        if ($this->title) {
            $meta[] = '<title>' . e($this->title) . '</title>';
        }

        if ($this->description) {
            $meta[] = '<meta name="description" content="' . e($this->description) . '">';
        }

        if ($this->keywords) {
            $meta[] = '<meta name="keywords" content="' . e($this->keywords) . '">';
        }

        if ($this->author) {
            $meta[] = '<meta name="author" content="' . e($this->author) . '">';
        }

        $meta[] = '<meta name="robots" content="' . e($this->robots) . '">';

        // Open Graph tags
        if ($this->title) {
            $meta[] = '<meta property="og:title" content="' . e($this->title) . '">';
        }

        if ($this->description) {
            $meta[] = '<meta property="og:description" content="' . e($this->description) . '">';
        }

        if ($this->url) {
            $meta[] = '<meta property="og:url" content="' . e($this->url) . '">';
        }

        if ($this->image) {
            $meta[] = '<meta property="og:image" content="' . e($this->image) . '">';
        }

        $meta[] = '<meta property="og:type" content="' . e($this->type) . '">';
        $meta[] = '<meta property="og:site_name" content="Phoenix Beverages Group">';

        // Additional Open Graph tags
        foreach ($this->openGraph as $property => $content) {
            $meta[] = '<meta property="' . e($property) . '" content="' . e($content) . '">';
        }

        // Twitter Card tags
        $meta[] = '<meta name="twitter:card" content="summary_large_image">';
        
        if ($this->title) {
            $meta[] = '<meta name="twitter:title" content="' . e($this->title) . '">';
        }

        if ($this->description) {
            $meta[] = '<meta name="twitter:description" content="' . e($this->description) . '">';
        }

        if ($this->image) {
            $meta[] = '<meta name="twitter:image" content="' . e($this->image) . '">';
        }

        // Additional Twitter Card tags
        foreach ($this->twitterCard as $property => $content) {
            $meta[] = '<meta name="' . e($property) . '" content="' . e($content) . '">';
        }

        // Canonical URL
        if ($this->canonical) {
            $meta[] = '<link rel="canonical" href="' . e($this->canonical) . '">';
        }

        return implode("\n    ", $meta);
    }

    public function renderJsonLd($data = [])
    {
        if (empty($data)) {
            return '';
        }

        return '<script type="application/ld+json">' . json_encode($data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>';
    }

    public function generateOrganizationSchema()
    {
        return $this->renderJsonLd([
            '@context' => 'https://schema.org',
            '@type' => 'Organization',
            'name' => 'Phoenix Beverages Group',
            'url' => config('app.url'),
            'logo' => asset('images/logo.png'),
            'description' => 'Phoenix Beverages Group is a leading beverage company in Mauritius and the Indian Ocean region.',
            'address' => [
                '@type' => 'PostalAddress',
                'addressCountry' => 'Mauritius'
            ],
            'contactPoint' => [
                '@type' => 'ContactPoint',
                'contactType' => 'customer service'
            ]
        ]);
    }

    public function generateBreadcrumbSchema($breadcrumbs)
    {
        $items = [];
        foreach ($breadcrumbs as $index => $breadcrumb) {
            $items[] = [
                '@type' => 'ListItem',
                'position' => $index + 1,
                'name' => $breadcrumb['name'],
                'item' => $breadcrumb['url'] ?? ''
            ];
        }

        return $this->renderJsonLd([
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => $items
        ]);
    }

    /**
     * Render structured data from database
     */
    public function renderStructuredData()
    {
        $output = '';
        
        foreach ($this->structuredData as $data) {
            if (isset($data['type']) && isset($data['data'])) {
                $schema = [
                    '@context' => 'https://schema.org',
                    '@type' => $data['type']
                ];
                
                foreach ($data['data'] as $property => $value) {
                    $schema[$property] = $value;
                }
                
                $output .= $this->renderJsonLd($schema) . "\n";
            }
        }
        
        return $output;
    }

    /**
     * Render hreflang tags for international SEO
     */
    public function renderHreflang()
    {
        $output = '';
        
        if (!empty($this->hreflang)) {
            foreach ($this->hreflang as $lang => $url) {
                $output .= '<link rel="alternate" hreflang="' . e($lang) . '" href="' . e($url) . '">' . "\n";
            }
        }
        
        return $output;
    }

    /**
     * Render alternate URLs for different languages/regions
     */
    public function renderAlternateUrls()
    {
        $output = '';
        
        if (!empty($this->alternateUrls)) {
            foreach ($this->alternateUrls as $url) {
                $output .= '<link rel="alternate" href="' . e($url['url']) . '" hreflang="' . e($url['lang']) . '">' . "\n";
            }
        }
        
        return $output;
    }

    /**
     * Get all SEO data as array for easy use in controllers
     */
    public function toArray()
    {
        return [
            'title' => $this->title,
            'description' => $this->description,
            'keywords' => $this->keywords,
            'image' => $this->image,
            'url' => $this->url,
            'type' => $this->type,
            'author' => $this->author,
            'robots' => $this->robots,
            'canonical' => $this->canonical,
            'open_graph' => $this->openGraph,
            'twitter_card' => $this->twitterCard,
            'structured_data' => $this->structuredData,
        ];
    }

    /**
     * Getter methods for accessing protected properties
     */
    public function getTitle() { return $this->title; }
    public function getDescription() { return $this->description; }
    public function getKeywords() { return $this->keywords; }
    public function getImage() { return $this->image; }
    public function getUrl() { return $this->url; }
    public function getType() { return $this->type; }
    public function getAuthor() { return $this->author; }
    public function getRobots() { return $this->robots; }
    public function getCanonical() { return $this->canonical; }
    public function getOpenGraph() { return $this->openGraph; }
    public function getTwitterCard() { return $this->twitterCard; }
    public function getStructuredData() { return $this->structuredData; }
    public function getHreflang() { return $this->hreflang; }
    public function getAlternateUrls() { return $this->alternateUrls; }

    /**
     * Set hreflang tags for international SEO
     */
    public function setHreflang($hreflang)
    {
        $this->hreflang = $hreflang;
        return $this;
    }

    /**
     * Set alternate URLs for different languages/regions
     */
    public function setAlternateUrls($alternateUrls)
    {
        $this->alternateUrls = $alternateUrls;
        return $this;
    }
} 