/**
 * DearFlip-like 3D PDF Flipbook Engine
 * Professional 3D flipbook with realistic page turning effects
 */

class DearFlipEngine {
    constructor() {
        this.instances = new Map();
        this.pdfRenderer = new PDFRenderer();
    }

    /**
     * Create a new 3D flipbook instance
     */
    async create(containerId, pdfPath, options = {}) {
        const container = document.getElementById(containerId);
        if (!container) {
            throw new Error(`Container ${containerId} not found`);
        }


        // Create flipbook container
        container.innerHTML = this.createFlipbookHTML(containerId);
        
        // Store instance FIRST, before loading PDF
        const instanceData = {
            container: container,
            pdfPath: pdfPath,
            options: options,
            currentPage: 1, // Will be adjusted based on actual page count
            totalPages: 0,
            pages: [],
            flipController: null, // For managing flip state
            isFlipping: false,
            flipStartTime: null,
            lastClickTime: null
        };
        
        this.instances.set(containerId, instanceData);
        
        // Load PDF and render pages
        await this.loadAndRenderPDF(container, pdfPath, options, containerId);
        
        // Initialize 3D effects
        this.initialize3DEffects(container, options);

        return this.instances.get(containerId);
    }

    /**
     * Defer expensive visibility updates to prevent layout thrashing
     */
    deferVisibilityUpdate(element, action) {
        requestAnimationFrame(() => {
            if (action === 'hide') {
                element.style.visibility = 'hidden';
                element.style.opacity = '0';
                // Defer display:none to next frame
                requestAnimationFrame(() => {
                    element.style.display = 'none';
                });
            } else if (action === 'show') {
                element.style.display = 'block';
                element.style.visibility = 'visible';
                element.style.opacity = '1';
            }
        });
    }

    /**
     * DearFlip-style: Batch DOM writes with requestAnimationFrame
     * Eliminates JS style writes mid-frame for smooth 60fps
     */
    batchStyleWrites(element, styles) {
        if (!element) return;
        
        requestAnimationFrame(() => {
            // Batch all style changes in a single frame
            // Only use GPU-accelerated properties to avoid reflow
            const gpuStyles = {};
            if (styles.transform) gpuStyles.transform = styles.transform;
            if (styles.opacity !== undefined) gpuStyles.opacity = styles.opacity;
            if (styles.visibility !== undefined) gpuStyles.visibility = styles.visibility;
            if (styles.zIndex !== undefined) gpuStyles.zIndex = styles.zIndex;
            if (styles.display !== undefined) gpuStyles.display = styles.display;
            if (styles.pointerEvents !== undefined) gpuStyles.pointerEvents = styles.pointerEvents;
            if (styles.backfaceVisibility !== undefined) gpuStyles.backfaceVisibility = styles.backfaceVisibility;
            if (styles.webkitBackfaceVisibility !== undefined) gpuStyles.webkitBackfaceVisibility = styles.webkitBackfaceVisibility;
            
            Object.assign(element.style, gpuStyles);
        });
    }

    /**
     * Minimal Enhancement: Spring physics easing
     * Adds realistic motion to existing flip animations
     */
    springEase(t) {
        return 1 - Math.pow(2, -10 * t) * Math.sin((t - 0.075) * (2 * Math.PI) / 0.3);
    }

    /**
     * Minimal Enhancement: Spine curvature effect
     * Adds subtle paper-like bending to transforms
     */
    addSpineCurvature(angle) {
        return Math.sin(angle * Math.PI / 180) * 2; // Subtle 2px curve
    }

    /**
     * Minimal Enhancement: Dynamic shadow update
     * Enhances existing page shadows with realistic gradients
     */
    updatePageShadow(pageShadow, progress, angle) {
        if (!pageShadow) return;
        
        const shadowOpacity = Math.min(0.8, progress * 1.2);
        const lightingIntensity = Math.sin(angle * Math.PI / 180) * 0.5 + 0.5;
        
        const shadowGradient = `linear-gradient(${angle + 90}deg, 
            rgba(0,0,0,${lightingIntensity * 0.3}) 0%, 
            rgba(0,0,0,${lightingIntensity * 0.1}) 50%, 
            transparent 100%)`;
        
        pageShadow.style.background = shadowGradient;
        pageShadow.style.opacity = shadowOpacity;
    }

    /**
     * Minimal Enhancement: Optional audio feedback
     * Adds subtle page turn sound without changing core logic
     */
    playPageTurnSound() {
        try {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            oscillator.frequency.setValueAtTime(200, audioContext.currentTime);
            oscillator.frequency.exponentialRampToValueAtTime(150, audioContext.currentTime + 0.2);
            
            gainNode.gain.setValueAtTime(0, audioContext.currentTime);
            gainNode.gain.linearRampToValueAtTime(0.05, audioContext.currentTime + 0.01);
            gainNode.gain.exponentialRampToValueAtTime(0.001, audioContext.currentTime + 0.2);
            
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.2);
        } catch (error) {
            // Silently fail if audio not available
        }
    }

    /**
     * Enhanced: Animate shadows during flip for realistic lighting
     * Updates shadow gradients based on flip progress
     */
    animateFlipShadows(flippingPage, direction) {
        const pageShadow = flippingPage.querySelector('.page-shadow');
        if (!pageShadow) return;

        const startTime = performance.now();
        const duration = 1200; // Match CSS transition duration
        
        const animate = (currentTime) => {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            // Calculate flip angle based on progress
            const angle = progress * 180 * (direction === 'forward' ? -1 : 1);
            
            // Update shadow with dynamic lighting
            this.updatePageShadow(pageShadow, progress, angle);
            
            // Continue animation if not complete
            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    }

    /**
     * Ensure page is pre-decoded and GPU-ready before flipping
     */
    /**
     * DearFlip-style: Pre-decode images for GPU texture upload
     * Ensures smooth animation without mid-frame texture uploads
     */
    async ensurePageReady(pageElement) {
        if (!pageElement) return;
        
        const img = pageElement.querySelector('img');
        if (!img) return;
        
        // Always pre-decode for GPU readiness (DearFlip-style)
        if (img.decode) {
            try {
                await img.decode();
                console.log('Page image pre-decoded for GPU');
                
                // Mark as GPU-ready
                pageElement.setAttribute('data-gpu-ready', 'true');
            } catch (error) {
                console.warn('Image decode failed, continuing anyway:', error);
            }
        } else if (!img.complete || img.naturalWidth === 0) {
            // Fallback for older browsers
            await new Promise((resolve, reject) => {
                const timeout = setTimeout(() => {
                    reject(new Error('Image load timeout'));
                }, 5000);
                
                img.onload = () => {
                    clearTimeout(timeout);
                    resolve();
                };
                img.onerror = () => {
                    clearTimeout(timeout);
                    reject(new Error('Image load failed'));
                };
            });
        }
    }

    /**
     * DearFlip-style: Update current spread for cache management
     */
    updateCurrentSpread(containerId) {
        const instance = this.instances.get(containerId);
        if (!instance) return;
        
        // Calculate current spread (2 pages per spread)
        const currentSpread = Math.ceil(instance.currentPage / 2);
        
        // Update PDF renderer's current spread for cache trimming
        if (this.pdfRenderer) {
            this.pdfRenderer.currentSpread = currentSpread;
            this.pdfRenderer.trimCache();
        }
        
        console.log(`Current spread updated: ${currentSpread}`);
    }

    /**
     * Handle zoom events for enhanced text sharpness
     */
    handleZoomEvent(zoomLevel) {
        if (this.pdfRenderer) {
            this.pdfRenderer.handleZoomEvent(zoomLevel);
            console.log(`Zoom event handled: ${zoomLevel}x`);
        }
    }

    /**
     * Set text sharpness mode for all instances
     */
    setTextSharpnessMode(mode) {
        if (this.pdfRenderer) {
            this.pdfRenderer.setTextSharpnessMode(mode);
            console.log(`Text sharpness mode set to: ${mode}`);
        }
    }

    /**
     * Enable/disable text layer overlay for all instances
     */
    setTextLayerEnabled(enabled) {
        if (this.pdfRenderer) {
            this.pdfRenderer.setTextLayerEnabled(enabled);
            console.log(`Text layer overlay ${enabled ? 'enabled' : 'disabled'}`);
        }
    }

    /**
     * DearFlip-style: Page pipeline scheduling
     * Overlap decode + animation during flips for seamless experience
     */
    async schedulePagePipeline(containerId, direction) {
        const instance = this.instances.get(containerId);
        if (!instance) return;
        
        const nextPages = direction === 'forward' 
            ? [instance.currentPage + 1, instance.currentPage + 2]
            : [instance.currentPage - 1, instance.currentPage - 2];
        
        // Stage 1: Start preloading immediately (async)
        this.preloadPages(containerId, nextPages);
        
        // Stage 2: During flip animation, ensure pages are ready
        const ensureReady = async () => {
            for (const pageNum of nextPages) {
                const pageEl = instance.container.querySelector(`[data-page-number="${pageNum}"]`);
                if (pageEl) {
                    await this.ensurePageReady(pageEl);
                }
            }
        };
        
        // Stage 3: By the time flip ends, pages are GPU-ready
        return ensureReady();
    }

    /**
     * DearFlip-style: Offscreen pre-compositing
     * Pre-render next spread offscreen with GPU compositing
     */
    createOffscreenPrecomposite(container, nextPages) {
        // Create offscreen container for pre-compositing
        const precompositeContainer = document.createElement('div');
        precompositeContainer.className = 'preload-spread';
        precompositeContainer.style.position = 'absolute';
        precompositeContainer.style.opacity = '0';
        precompositeContainer.style.pointerEvents = 'none';
        precompositeContainer.style.willChange = 'transform, opacity';
        precompositeContainer.style.transform = 'translateZ(0)';
        
        // Add next pages to offscreen container
        nextPages.forEach((pageNum, index) => {
            const pageEl = container.querySelector(`[data-page-number="${pageNum}"]`);
            if (pageEl) {
                const clone = pageEl.cloneNode(true);
                clone.style.position = 'absolute';
                clone.style.width = '50%';
                clone.style.height = '100%';
                clone.style.left = index === 0 ? '0' : '50%';
                precompositeContainer.appendChild(clone);
            }
        });
        
        container.appendChild(precompositeContainer);
        return precompositeContainer;
    }

    /**
     * DearFlip-style: Ultimate smoothness integration
     * Combines all advanced optimizations for maximum performance
     */
    async ultimateSmoothFlip(container, direction) {
        const instance = this.instances.get(container.id);
        if (!instance) return;

        try {
            // Stage 1: Start page pipeline scheduling immediately
            const pipelinePromise = this.schedulePagePipeline(container.id, direction);
            
            // Stage 2: Create offscreen pre-composite
            const nextPages = direction === 'forward' 
                ? [instance.currentPage + 1, instance.currentPage + 2]
                : [instance.currentPage - 1, instance.currentPage - 2];
            
            const precomposite = this.createOffscreenPrecomposite(container, nextPages);
            
            // Stage 3: Start flip animation with pure CSS transforms
            const flipPromise = this.startPureCSSFlip(container, direction);
            
            // Stage 4: Wait for pipeline to complete
            await pipelinePromise;
            
            // Stage 5: Reveal pre-composite when ready
            requestAnimationFrame(() => {
                precomposite.style.opacity = '1';
                precomposite.style.transition = 'opacity 0.3s ease-in-out';
            });
            
            // Stage 6: Wait for flip to complete
            await flipPromise;
            
            // Stage 7: Clean up pre-composite
            setTimeout(() => {
                if (precomposite.parentNode) {
                    precomposite.parentNode.removeChild(precomposite);
                }
            }, 300);
            
        } catch (error) {
            console.error('Ultimate smooth flip error:', error);
        }
    }

    /**
     * Single Page Flip Animation
     * Flips only one page at a time for natural feel
     */
   /**
 * Smooth Single Page Flip Animation
 * Flips only one page at a time, with natural easing and cleanup.
 */
async startSinglePageFlip(container, direction = 'forward') {
    const leftPage = container.querySelector('.current-left');
    const rightPage = container.querySelector('.current-right');
    if (!leftPage || !rightPage) return;
  
    const flippingPage = direction === 'forward' ? rightPage : leftPage;
  
    // 🟩 Phase 0: GPU warm-up
    flippingPage.style.willChange = 'transform, opacity';
    flippingPage.style.transform = 'translateZ(0)';
  
    // 🟦 Phase 1: Initiate Flip
    flippingPage.classList.add('page-flip');
    flippingPage.style.transition = 'transform 1s cubic-bezier(0.25, 0.8, 0.25, 1)';
  
    // Set rotation based on direction
    const angle = direction === 'forward' ? -180 : 180;
    requestAnimationFrame(() => {
      flippingPage.style.transform = `rotateY(${angle}deg)`;
    });
  
    console.log(`Single page flip: Flipping ${direction === 'forward' ? 'right' : 'left'} page`);
  
    // 🟨 Phase 2–3: Wait for transition
    await new Promise((resolve) => {
      const onEnd = (e) => {
        if (e.target !== flippingPage) return;
        flippingPage.removeEventListener('transitionend', onEnd);
        resolve();
      };
      flippingPage.addEventListener('transitionend', onEnd);
    });
  
    // 🟧 Phase 4: Cleanup + Reset
    flippingPage.style.willChange = 'auto';
    flippingPage.classList.remove('page-flip', 'reverse');
    flippingPage.style.transition = '';
    flippingPage.style.transform = '';
    
    // Optional: hide or reorder layers
    flippingPage.style.zIndex = direction === 'forward' ? 1 : 5;
  
    console.log('Flip complete');
  }
  

    /**
     * Smooth reveal animation using requestAnimationFrame
     */
    smoothReveal(elements, duration = 600) {
        let start;
        const animate = (timestamp) => {
            if (!start) start = timestamp;
            const progress = Math.min((timestamp - start) / duration, 1);
            const eased = 1 - Math.pow(1 - progress, 1.5); // Gentle ease-out
            
            elements.forEach(({ element, startOpacity, endOpacity }) => {
                const opacity = startOpacity + (endOpacity - startOpacity) * eased;
                element.style.opacity = opacity;
            });
            
            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };
        requestAnimationFrame(animate);
    }

    /**
     * Create flipbook HTML structure
     */
    createFlipbookHTML(containerId) {
        return `
            <style>
                .dearflip-container {
            width: 92%;
            max-width: 1350px;
            height: 93vh;
            position: relative;
            perspective: 1500px;
            perspective-origin: center center;
            background: linear-gradient(135deg, 
                rgba(30, 30, 30, 0.95) 0%, 
                rgba(20, 20, 20, 0.9) 50%, 
                rgba(30, 30, 30, 0.95) 100%);
            border-radius: 0px;
            overflow: visible;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            box-shadow: 
                0 30px 60px rgba(0, 0, 0, 0.25),
                0 15px 30px rgba(0, 0, 0, 0.15),
                inset 0 2px 4px rgba(255, 255, 255, 0.1);
            margin: 2vh auto;
            border: 2px solid rgba(15, 15, 15, 0.8);
            padding: 20px;
            transition: transform 0.1s ease-out;
        }
        
        .dearflip-container.draggable {
            cursor: grab;
        }
        
        .dearflip-container.dragging {
            cursor: grabbing;
        }

        /* DearFlip-style: Fixed 3D scene container for persistent GPU context */
        .flip-scene {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            perspective: 1500px;
            transform-style: preserve-3d;
            will-change: transform;
            contain: layout paint size style;
            overflow: hidden;
        }

        .flip-page {
            position: absolute;
            width: 50%;
            height: 100%;
            backface-visibility: hidden;
            will-change: transform, opacity;
            transform: translateZ(0);
            transform-style: preserve-3d;
        }

        .flip-page.left {
            left: 0;
            transform-origin: right center;
        }

        .flip-page.right {
            right: 0;
            transform-origin: left center;
        }

        /* Offscreen pre-compositing (DearFlip-style) */
        .preload-spread {
            opacity: 0;
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            will-change: transform, opacity;
            transform: translateZ(0);
            contain: layout paint size style;
        }

        /* DearFlip-style: Browser optimization flags */
        .dearflip-container {
            contain: layout paint size style;
            overflow: hidden;
            isolation: isolate;
            transform: translateZ(0);
        }

        .dearflip-page {
            contain: layout paint size style;
            isolation: isolate;
        }

        /* GPU layer optimization */
        .dearflip-page,
        .page-flip,
        .flip-page {
            transform: translateZ(0);
            backface-visibility: hidden;
            -webkit-backface-visibility: hidden;
        }
        
        .dearflip-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: 
                /* Top-left diagonal line */
                linear-gradient(135deg, transparent 0%, transparent 48%, rgba(0, 0, 0, 0.4) 49%, rgba(0, 0, 0, 0.4) 51%, transparent 52%),
                /* Top-right diagonal line */
                linear-gradient(225deg, transparent 0%, transparent 48%, rgba(0, 0, 0, 0.4) 49%, rgba(0, 0, 0, 0.4) 51%, transparent 52%),
                /* Bottom-left diagonal line */
                linear-gradient(45deg, transparent 0%, transparent 48%, rgba(0, 0, 0, 0.4) 49%, rgba(0, 0, 0, 0.4) 51%, transparent 52%),
                /* Bottom-right diagonal line */
                linear-gradient(315deg, transparent 0%, transparent 48%, rgba(0, 0, 0, 0.4) 49%, rgba(0, 0, 0, 0.4) 51%, transparent 52%);
            pointer-events: none;
            z-index: 1;
        }
        
        .book-content {
            width: 100%;
            height: 100%;
            background: linear-gradient(135deg, 
                rgba(248, 249, 250, 0.98) 0%, 
                rgba(255, 255, 255, 1) 50%, 
                rgba(248, 249, 250, 0.98) 100%);
            border-radius: 0px;
            position: relative;
            box-shadow: 
                0 15px 30px rgba(0, 0, 0, 0.2),
                0 8px 16px rgba(0, 0, 0, 0.1),
                inset 0 1px 0 rgba(255, 255, 255, 0.8);
            border: 1px solid rgba(200, 200, 200, 0.3);
            overflow: hidden;
        }
                
                /* Beautiful Loading State */
                .dearflip-loader {
                    width: 100%;
                    height: 100vh;
                    position: relative;
                    background: linear-gradient(135deg, 
                        rgba(10, 31, 46, 0.9) 0%, 
                        rgba(8, 186, 187, 0.08) 50%, 
                        rgba(10, 31, 46, 0.9) 100%);
                    display: flex;
                    flex-direction: column;
                    align-items: center;
                    justify-content: center;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                }
                
                .loader-content {
                    text-align: center;
                    color: #0A1F2E;
                    z-index: 10;
                }
                
                .loader-title {
                    font-size: 2.5rem;
                    font-weight: 300;
                    margin-bottom: 1rem;
                    opacity: 0.9;
                    letter-spacing: 2px;
                }
                
                .loader-subtitle {
                    font-size: 1.1rem;
                    margin-bottom: 3rem;
                    opacity: 0.7;
                    font-weight: 400;
                }
                
                /* Animated Book Icon */
                .book-loader {
                    width: 80px;
                    height: 100px;
                    position: relative;
                    margin: 0 auto 2rem;
                    transform-style: preserve-3d;
                    animation: bookFloat 3s ease-in-out infinite;
                }
                
                .book-loader::before {
                    content: '';
                    position: absolute;
                    width: 100%;
                    height: 100%;
                    background: linear-gradient(135deg, 
                        #0A1F2E 0%, 
                        #08BABB 50%, 
                        #0A1F2E 100%);
                    border-radius: 8px 4px 4px 8px;
                    box-shadow: 
                        0 0 0 2px rgba(8, 186, 187, 0.3),
                        0 8px 25px rgba(8, 186, 187, 0.2);
                    transform: rotateY(-15deg);
                }
                
                .book-loader::after {
                    content: '';
                    position: absolute;
                    width: 100%;
                    height: 100%;
                    background: linear-gradient(135deg, 
                        #08BABB 0%, 
                        #0A1F2E 50%, 
                        #08BABB 100%);
                    border-radius: 4px 8px 8px 4px;
                    box-shadow: 
                        0 0 0 2px rgba(10, 31, 46, 0.3),
                        0 8px 25px rgba(10, 31, 46, 0.2);
                    transform: rotateY(15deg);
                }
                
                @keyframes bookFloat {
                    0%, 100% { 
                        transform: translateY(0px) rotateX(5deg); 
                    }
                    50% { 
                        transform: translateY(-10px) rotateX(-5deg); 
                    }
                }
                
                /* Spinning Pages Animation */
                .page-spinner {
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%);
                    width: 60px;
                    height: 80px;
                    z-index: 5;
                }
                
                .page-spinner::before,
                .page-spinner::after {
                    content: '';
                    position: absolute;
                    width: 30px;
                    height: 80px;
                    background: linear-gradient(90deg, 
                        rgba(255, 255, 255, 0.9) 0%, 
                        rgba(255, 255, 255, 0.7) 50%, 
                        rgba(255, 255, 255, 0.9) 100%);
                    border-radius: 2px;
                    transform-origin: left center;
                    animation: pageFlip 2s ease-in-out infinite;
                }
                
                .page-spinner::before {
                    left: 0;
                    animation-delay: 0s;
                }
                
                .page-spinner::after {
                    right: 0;
                    animation-delay: 1s;
                }
                
                @keyframes pageFlip {
                    0%, 100% { 
                        transform: rotateY(0deg); 
                        opacity: 0.8;
                    }
                    50% { 
                        transform: rotateY(-90deg); 
                        opacity: 0.4;
                    }
                }
                
                /* Progress Bar */
                .loader-progress {
                    width: 300px;
                    height: 4px;
                    background: rgba(8, 186, 187, 0.15);
                    border-radius: 2px;
                    overflow: hidden;
                    margin-top: 2rem;
                }
                
                .loader-progress-bar {
                    height: 100%;
                    background: linear-gradient(90deg, 
                        #0A1F2E 0%, 
                        #08BABB 50%, 
                        #0A1F2E 100%);
                    border-radius: 2px;
                    animation: progressLoad 3s ease-in-out infinite;
                }
                
                @keyframes progressLoad {
                    0% { 
                        width: 0%; 
                        opacity: 0.6;
                    }
                    50% { 
                        width: 70%; 
                        opacity: 1;
                    }
                    100% { 
                        width: 100%; 
                        opacity: 0.6;
                    }
                }
                
                /* Loading Text Animation */
                .loading-text {
                    margin-top: 1.5rem;
                    font-size: 0.9rem;
                    opacity: 0.8;
                    animation: textPulse 2s ease-in-out infinite;
                }
                
                @keyframes textPulse {
                    0%, 100% { 
                        opacity: 0.6; 
                    }
                    50% { 
                        opacity: 1; 
                    }
                }
                
                /* Real book binding - seamless join with subtle center curve */
                .book-spine {
                    position: absolute;
                    left: 50%;
                    top: 0;
                    bottom: 0;
                    width: 2px;
                    background: linear-gradient(
                        to bottom,
                        rgba(0, 0, 0, 0.1) 0%,
                        rgba(0, 0, 0, 0.15) 20%,
                        rgba(0, 0, 0, 0.2) 50%,
                        rgba(0, 0, 0, 0.15) 80%,
                        rgba(0, 0, 0, 0.1) 100%
                    );
                    transform: translateX(-50%);
                    z-index: 5;
                    box-shadow: 
                        inset 1px 0 2px rgba(0, 0, 0, 0.1),
                        inset -1px 0 2px rgba(0, 0, 0, 0.1);
                }
                
                /* Add subtle curve effect at the center where pages meet */
                .book-spine::before {
                    content: '';
                    position: absolute;
                    top: 0;
                    left: -1px;
                    right: -1px;
                    bottom: 0;
                    background: linear-gradient(
                        to right,
                        transparent 0%,
                        rgba(0, 0, 0, 0.05) 20%,
                        rgba(0, 0, 0, 0.08) 50%,
                        rgba(0, 0, 0, 0.05) 80%,
                        transparent 100%
                    );
                    border-radius: 1px;
                }
                
                /* Add subtle inward curve effect to pages at the center */
                .dearflip-page.current-left::after {
                    content: '';
                    position: absolute;
                    top: 0;
                    right: 0;
                    width: 3px;
                    height: 100%;
                    background: linear-gradient(
                        to right,
                        rgba(0, 0, 0, 0.1) 0%,
                        rgba(0, 0, 0, 0.15) 50%,
                        rgba(0, 0, 0, 0.1) 100%
                    );
                    border-radius: 0 2px 2px 0;
                    box-shadow: 
                        inset 1px 0 2px rgba(0, 0, 0, 0.1),
                        1px 0 3px rgba(0, 0, 0, 0.05);
                }
                
                .dearflip-page.current-right::after {
                    content: '';
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 3px;
                    height: 100%;
                    background: linear-gradient(
                        to left,
                        rgba(0, 0, 0, 0.1) 0%,
                        rgba(0, 0, 0, 0.15) 50%,
                        rgba(0, 0, 0, 0.1) 100%
                    );
                    border-radius: 2px 0 0 2px;
                    box-shadow: 
                        inset -1px 0 2px rgba(0, 0, 0, 0.1),
                        -1px 0 3px rgba(0, 0, 0, 0.05);
                }
                
                .dearflip-pages {
                    width: 100%;
                    height: 100%;
                    position: relative;
                    transform-style: preserve-3d;
                    display: flex;
                    align-items: flex-start;
                    justify-content: center;
                    z-index: 2;
                    padding-top: 2.5%;
                }
                
                .dearflip-page {
                    position: absolute;
                    width: 50%;
                    height: 98%;
                    top: 1%;
                    background: 
                        /* Paper texture */
                        radial-gradient(circle at 20% 20%, rgba(255, 255, 255, 0.8) 0%, transparent 50%),
                        radial-gradient(circle at 80% 80%, rgba(255, 255, 255, 0.6) 0%, transparent 50%),
                        radial-gradient(circle at 40% 60%, rgba(255, 255, 255, 0.4) 0%, transparent 50%),
                        /* Base paper color */
                        linear-gradient(135deg, #fefefe 0%, #f8f8f8 100%);
                    transform-style: preserve-3d;
                    transform-origin: center center;
                    transition: all 1.2s cubic-bezier(0.25, 0.46, 0.45, 0.94);
                    cursor: pointer;
                    overflow: hidden;
                    border-radius: 4px;
                    box-shadow: 
                        /* Realistic book page shadows */
                        0 10px 35px rgba(0, 0, 0, 0.13),
                        0 5px 18px rgba(0, 0, 0, 0.09),
                        0 2px 8px rgba(0, 0, 0, 0.07),
                        /* Inner page texture */
                        inset 0 1px 0 rgba(255, 255, 255, 0.9),
                        inset 0 -1px 0 rgba(0, 0, 0, 0.08),
                        /* Center binding shadow */
                        inset 1px 0 2px rgba(0, 0, 0, 0.04);
                    backface-visibility: hidden;
                    -webkit-backface-visibility: hidden;
                    border: 1px solid rgba(0, 0, 0, 0.05);
                    padding: 0;
                    margin: 0;
                    will-change: transform, opacity;
                    translate: 0; /* triggers GPU pipeline */
                }
                
                .dearflip-page img {
                    width: 100%;
                    height: 100%;
                    object-fit: contain;
                    object-position: center;
                    display: block;
                    margin: 0;
                    padding: 0;
                    border: none;
                    outline: none;
                    /* Enhanced text clarity and image quality for PDF readability */
                    image-rendering: -webkit-optimize-contrast;
                    image-rendering: crisp-edges;
                    image-rendering: high-quality;
                    /* Use auto for best browser optimization of high-res images */
                    image-rendering: auto;
                    -webkit-font-smoothing: antialiased;
                    -moz-osx-font-smoothing: grayscale;
                    text-rendering: optimizeLegibility;
                    /* Prevent any blurring filters */
                    filter: none;
                    transform: translateZ(0); /* GPU acceleration */
                }
                
                .dearflip-page.left-page {
                    left: 0%;
                    transform: rotateY(-5deg) translateZ(-3px) translateX(-1px);
                    transform-origin: right center;
                    z-index: 1;
                    opacity: 0.6;
                    filter: brightness(0.9);
                }
                
                .dearflip-page.right-page {
                    right: 0%;
                    transform: rotateY(5deg) translateZ(-3px) translateX(1px);
                    transform-origin: left center;
                    z-index: 1;
                    opacity: 0.6;
                    filter: brightness(0.9);
                }
                
                .dearflip-page.current-left {
                    left: 0%;
                    transform: rotateY(-0.5deg) translateZ(0px) translateX(-0.5px);
                    transform-origin: right center;
                    z-index: 5;
                    opacity: 1;
                    filter: brightness(1);
                }
                
                .dearflip-page.current-right {
                    right: 0%;
                    transform: rotateY(0.5deg) translateZ(0px) translateX(0.5px);
                    transform-origin: left center;
                    z-index: 5;
                    opacity: 1;
                    filter: brightness(1);
                }
                
                .dearflip-page img {
                    width: 100%;
                    height: 100%;
                    object-fit: contain;
                    border-radius: 3px;
                    /* Enhanced text clarity and image quality for PDF readability */
                    image-rendering: -webkit-optimize-contrast;
                    image-rendering: crisp-edges;
                    image-rendering: high-quality;
                    /* Use auto for best browser optimization of high-res images */
                    image-rendering: auto;
                    -webkit-font-smoothing: antialiased;
                    -moz-osx-font-smoothing: grayscale;
                    text-rendering: optimizeLegibility;
                    /* Prevent any blurring filters */
                    filter: none;
                    transform: translateZ(0); /* GPU acceleration */
                }
                
                /* Ensure pages are visible and properly positioned */
                .dearflip-page[style*="display: block"] {
                    display: block !important;
                }
                
                .dearflip-page[style*="visibility: visible"] {
                    visibility: visible !important;
                }
                
                /* Ensure proper layering and prevent background pages from showing */
                .dearflip-page {
                    backface-visibility: hidden;
                    -webkit-backface-visibility: hidden;
                }
                
                .dearflip-page[style*="opacity: 0"] {
                    opacity: 0 !important;
                    pointer-events: none;
                }
                
                .dearflip-page[style*="opacity: 1"] {
                    opacity: 1 !important;
                    pointer-events: auto;
                }
                
                .dearflip-page:hover {
                    transform: translateZ(4px);
                    box-shadow: 
                        /* Realistic book page shadows */
                        0 12px 40px rgba(0, 0, 0, 0.15),
                        0 6px 20px rgba(0, 0, 0, 0.1),
                        0 3px 10px rgba(0, 0, 0, 0.08),
                        /* Inner page texture */
                        inset 0 1px 0 rgba(255, 255, 255, 0.9),
                        inset 0 -1px 0 rgba(0, 0, 0, 0.08),
                        /* Center binding shadow */
                        inset 1px 0 3px rgba(0, 0, 0, 0.05);
                }
                
                /* Smooth page state transitions */
                .current-left, .current-right {
                    transition: all 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
                    will-change: transform, opacity;
                }
                
                .left-page, .right-page {
                    transition: all 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
                    will-change: transform, opacity;
                }
                
                /* Enhanced page visibility during transitions */
                .dearflip-page img {
                    transition: opacity 0.8s ease-in-out;
                    will-change: opacity;
                }
                
                .page-flip {
                    /* Simplified: Basic flip without complex effects */
                    will-change: transform;
                    transform-style: preserve-3d;
                    backface-visibility: hidden;
                    transition: transform 0.8s ease-out;
                    transform-origin: left center;
                    z-index: 20;
                    opacity: 1 !important;
                    visibility: visible !important;
                    display: block !important;
                }
                
                
                
                .page-flip.reverse {
                    /* Simplified: Basic reverse flip without complex effects */
                    will-change: transform;
                    transform-style: preserve-3d;
                    backface-visibility: hidden;
                    transition: transform 0.8s ease-out;
                    transform-origin: right center;
                    z-index: 20;
                    opacity: 1 !important;
                    visibility: visible !important;
                    display: block !important;
                }
                
                /* Ensure backface content is visible during flip */
                .page-flip > *,
                .page-flip.reverse > * {
                    backface-visibility: visible !important;
                    -webkit-backface-visibility: visible !important;
                    opacity: 1 !important;
                    visibility: visible !important;
                    display: block !important;
                }
                
                /* Ensure images in backface content are visible */
                .page-flip img,
                .page-flip.reverse img {
                    backface-visibility: visible !important;
                    -webkit-backface-visibility: visible !important;
                    opacity: 1 !important;
                    visibility: visible !important;
                    display: block !important;
                }
                
                /* Ensure all backface content is visible during flip */
                .page-flip [style*="rotateY(180deg)"],
                .page-flip.reverse [style*="rotateY(180deg)"] {
                    backface-visibility: visible !important;
                    -webkit-backface-visibility: visible !important;
                    opacity: 1 !important;
                    visibility: visible !important;
                    display: block !important;
                }
                
                .page-flip [style*="rotateY(180deg)"] img,
                .page-flip.reverse [style*="rotateY(180deg)"] img {
                    backface-visibility: visible !important;
                    -webkit-backface-visibility: visible !important;
                    opacity: 1 !important;
                    visibility: visible !important;
                    display: block !important;
                }
                
                /* Ensure original content stays visible during flip */
                .page-flip > img,
                .page-flip.reverse > img {
                    backface-visibility: visible !important;
                    -webkit-backface-visibility: visible !important;
                    opacity: 1 !important;
                    visibility: visible !important;
                    display: block !important;
                }
                
                /* Hide backface content initially */
                .page-flip [style*="rotateY(180deg)"][style*="opacity: 0"],
                .page-flip.reverse [style*="rotateY(180deg)"][style*="opacity: 0"] {
                    opacity: 0 !important;
                    visibility: hidden !important;
                }
                
                
                
                
                /* Complete realistic book page flip animation */
                @keyframes pageFlip {
                    0% { 
                        transform: rotateY(0deg) translateZ(0px) translateX(0px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    2% { 
                        transform: rotateY(-3deg) translateZ(1px) translateX(-1px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    5% { 
                        transform: rotateY(-8deg) translateZ(2px) translateX(-2px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    10% { 
                        transform: rotateY(-15deg) translateZ(4px) translateX(-3px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    15% { 
                        transform: rotateY(-25deg) translateZ(6px) translateX(-4px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    20% { 
                        transform: rotateY(-35deg) translateZ(8px) translateX(-5px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    25% { 
                        transform: rotateY(-45deg) translateZ(10px) translateX(-6px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    30% { 
                        transform: rotateY(-55deg) translateZ(12px) translateX(-7px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    35% { 
                        transform: rotateY(-65deg) translateZ(14px) translateX(-8px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    40% { 
                        transform: rotateY(-75deg) translateZ(16px) translateX(-9px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    45% { 
                        transform: rotateY(-85deg) translateZ(18px) translateX(-10px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    50% { 
                        transform: rotateY(-90deg) translateZ(20px) translateX(-10px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    55% { 
                        transform: rotateY(-95deg) translateZ(18px) translateX(-9px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    60% { 
                        transform: rotateY(-105deg) translateZ(16px) translateX(-8px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    65% { 
                        transform: rotateY(-115deg) translateZ(14px) translateX(-7px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    70% { 
                        transform: rotateY(-125deg) translateZ(12px) translateX(-6px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    75% { 
                        transform: rotateY(-135deg) translateZ(10px) translateX(-5px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    80% { 
                        transform: rotateY(-145deg) translateZ(8px) translateX(-4px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    85% { 
                        transform: rotateY(-155deg) translateZ(6px) translateX(-3px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    90% { 
                        transform: rotateY(-165deg) translateZ(4px) translateX(-2px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    90% { 
                        transform: rotateY(-170deg) translateZ(4px) translateX(-2px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    93% { 
                        transform: rotateY(-175deg) translateZ(2px) translateX(-1px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    96% { 
                        transform: rotateY(-178deg) translateZ(1px) translateX(-0.5px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    98% { 
                        transform: rotateY(-179.5deg) translateZ(0.5px) translateX(-0.2px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    99% { 
                        transform: rotateY(-179.8deg) translateZ(0.2px) translateX(-0.1px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    100% { 
                        transform: rotateY(-180deg) translateZ(0px) translateX(0px);
                        opacity: 1;
                        z-index: 1;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                }
                
                @keyframes pageFlipReverse {
                    0% { 
                        transform: rotateY(0deg) translateZ(0px) translateX(0px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    2% { 
                        transform: rotateY(3deg) translateZ(1px) translateX(1px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    5% { 
                        transform: rotateY(8deg) translateZ(2px) translateX(2px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    10% { 
                        transform: rotateY(15deg) translateZ(4px) translateX(3px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    15% { 
                        transform: rotateY(25deg) translateZ(6px) translateX(4px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    20% { 
                        transform: rotateY(35deg) translateZ(8px) translateX(5px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    25% { 
                        transform: rotateY(45deg) translateZ(10px) translateX(6px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    30% { 
                        transform: rotateY(55deg) translateZ(12px) translateX(7px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    35% { 
                        transform: rotateY(65deg) translateZ(14px) translateX(8px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    40% { 
                        transform: rotateY(75deg) translateZ(16px) translateX(9px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    45% { 
                        transform: rotateY(85deg) translateZ(18px) translateX(10px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    50% { 
                        transform: rotateY(90deg) translateZ(20px) translateX(10px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    55% { 
                        transform: rotateY(95deg) translateZ(18px) translateX(9px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    60% { 
                        transform: rotateY(105deg) translateZ(16px) translateX(8px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    65% { 
                        transform: rotateY(115deg) translateZ(14px) translateX(7px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    70% { 
                        transform: rotateY(125deg) translateZ(12px) translateX(6px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    75% { 
                        transform: rotateY(135deg) translateZ(10px) translateX(5px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    80% { 
                        transform: rotateY(145deg) translateZ(8px) translateX(4px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    85% { 
                        transform: rotateY(155deg) translateZ(6px) translateX(3px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    90% { 
                        transform: rotateY(170deg) translateZ(4px) translateX(2px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    93% { 
                        transform: rotateY(175deg) translateZ(2px) translateX(1px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    96% { 
                        transform: rotateY(178deg) translateZ(1px) translateX(0.5px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    98% { 
                        transform: rotateY(179.5deg) translateZ(0.5px) translateX(0.2px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    99% { 
                        transform: rotateY(179.8deg) translateZ(0.2px) translateX(0.1px);
                        opacity: 1;
                        z-index: 20;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                    100% { 
                        transform: rotateY(180deg) translateZ(0px) translateX(0px);
                        opacity: 1;
                        z-index: 1;
                        visibility: visible;
                        display: block;
                        backface-visibility: visible;
                    }
                }
                
                /* Enhanced page shadow during next flip */
                .page-flip .page-shadow {
                    opacity: 1;
                    background: linear-gradient(90deg, 
                        rgba(0, 0, 0, 0.2) 0%, 
                        rgba(0, 0, 0, 0.12) 20%, 
                        rgba(0, 0, 0, 0.06) 40%, 
                        rgba(0, 0, 0, 0.02) 60%, 
                        rgba(0, 0, 0, 0.06) 80%, 
                        rgba(0, 0, 0, 0.15) 100%);
                    animation: pageShadowNext 1.5s cubic-bezier(0.25, 0.46, 0.45, 0.94) forwards;
                }
                
                @keyframes pageShadowNext {
                    0% { 
                        opacity: 0.3; 
                        transform: scaleX(1) scaleY(1);
                    }
                    20% { 
                        opacity: 0.6; 
                        transform: scaleX(1.02) scaleY(1.01);
                    }
                    40% { 
                        opacity: 0.8; 
                        transform: scaleX(1.05) scaleY(1.02);
                    }
                    60% { 
                        opacity: 0.9; 
                        transform: scaleX(1.08) scaleY(1.03);
                    }
                    80% { 
                        opacity: 0.7; 
                        transform: scaleX(1.05) scaleY(1.02);
                    }
                    100% { 
                        opacity: 0.4; 
                        transform: scaleX(1) scaleY(1);
                    }
                }
                
                /* Enhanced page curl effect for next flip */
                .page-flip::before {
                    content: '';
                    position: absolute;
                    top: 0;
                    right: 0;
                    width: 25px;
                    height: 100%;
                    background: linear-gradient(90deg, 
                        rgba(0, 0, 0, 0.15) 0%, 
                        rgba(0, 0, 0, 0.08) 30%, 
                        rgba(0, 0, 0, 0.03) 60%, 
                        transparent 100%);
                    transform-origin: right center;
                    animation: pageCurlNext 1.5s cubic-bezier(0.25, 0.46, 0.45, 0.94) forwards;
                    border-radius: 0 2px 2px 0;
                }
                
                @keyframes pageCurlNext {
                    0% { 
                        transform: rotateY(0deg) scaleY(1); 
                        opacity: 0; 
                        width: 25px;
                    }
                    5% { 
                        transform: rotateY(-5deg) scaleY(1.01); 
                        opacity: 0.1; 
                        width: 24px;
                    }
                    10% { 
                        transform: rotateY(-10deg) scaleY(1.02); 
                        opacity: 0.2; 
                        width: 23px;
                    }
                    15% { 
                        transform: rotateY(-18deg) scaleY(1.03); 
                        opacity: 0.3; 
                        width: 22px;
                    }
                    20% { 
                        transform: rotateY(-25deg) scaleY(1.04); 
                        opacity: 0.4; 
                        width: 21px;
                    }
                    25% { 
                        transform: rotateY(-35deg) scaleY(1.05); 
                        opacity: 0.5; 
                        width: 20px;
                    }
                    30% { 
                        transform: rotateY(-45deg) scaleY(1.06); 
                        opacity: 0.6; 
                        width: 19px;
                    }
                    35% { 
                        transform: rotateY(-55deg) scaleY(1.07); 
                        opacity: 0.7; 
                        width: 18px;
                    }
                    40% { 
                        transform: rotateY(-65deg) scaleY(1.08); 
                        opacity: 0.8; 
                        width: 17px;
                    }
                    45% { 
                        transform: rotateY(-75deg) scaleY(1.09); 
                        opacity: 0.9; 
                        width: 16px;
                    }
                    50% { 
                        transform: rotateY(-85deg) scaleY(1.1); 
                        opacity: 1; 
                        width: 15px;
                    }
                    55% { 
                        transform: rotateY(-95deg) scaleY(1.09); 
                        opacity: 0.9; 
                        width: 14px;
                    }
                    60% { 
                        transform: rotateY(-105deg) scaleY(1.08); 
                        opacity: 0.8; 
                        width: 13px;
                    }
                    65% { 
                        transform: rotateY(-115deg) scaleY(1.07); 
                        opacity: 0.7; 
                        width: 12px;
                    }
                    70% { 
                        transform: rotateY(-125deg) scaleY(1.06); 
                        opacity: 0.6; 
                        width: 11px;
                    }
                    75% { 
                        transform: rotateY(-135deg) scaleY(1.05); 
                        opacity: 0.5; 
                        width: 10px;
                    }
                    80% { 
                        transform: rotateY(-145deg) scaleY(1.04); 
                        opacity: 0.4; 
                        width: 8px;
                    }
                    85% { 
                        transform: rotateY(-155deg) scaleY(1.03); 
                        opacity: 0.3; 
                        width: 6px;
                    }
                    90% { 
                        transform: rotateY(-165deg) scaleY(1.02); 
                        opacity: 0.2; 
                        width: 4px;
                    }
                    95% { 
                        transform: rotateY(-175deg) scaleY(1.01); 
                        opacity: 0.1; 
                        width: 2px;
                    }
                    100% { 
                        transform: rotateY(-180deg) scaleY(1); 
                        opacity: 0; 
                        width: 0px;
                    }
                }
                
                .page-shadow {
                    position: absolute;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    background: linear-gradient(90deg, 
                        rgba(0, 0, 0, 0.08) 0%, 
                        rgba(0, 0, 0, 0.03) 50%, 
                        rgba(0, 0, 0, 0.08) 100%);
                    border-radius: 3px;
                    pointer-events: none;
                    opacity: 0;
                    transition: opacity 0.3s ease, background 0.1s ease;
                    will-change: opacity, background;
                }
                
                .dearflip-page.flipping .page-shadow {
                    opacity: 1;
                }
                
                .close-btn {
                    position: absolute;
                    top: 20px;
                    right: 20px;
                    width: 40px;
                    height: 40px;
                    background: rgba(0, 0, 0, 0.6);
                    color: white;
                    border: none;
                    border-radius: 50%;
                    cursor: pointer;
                    font-size: 20px;
                    font-weight: bold;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    z-index: 20;
                    transition: all 0.3s ease;
                }
                
                .close-btn:hover {
                    background: rgba(0, 0, 0, 0.8);
                    transform: scale(1.1);
                }
                
                
                .close-btn {
                    position: absolute;
                    top: 20px;
                    right: 20px;
                    width: 40px;
                    height: 40px;
                    background: rgba(0, 0, 0, 0.6);
                    border: none;
                    border-radius: 50%;
                    color: white;
                    font-size: 18px;
                    cursor: pointer;
                    transition: all 0.3s ease;
                    z-index: 100;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }
                
                .close-btn:hover {
                    background: rgba(0, 0, 0, 0.8);
                    transform: scale(1.1);
                }
            </style>
            
            <div class="dearflip-loader" id="dearflip-loader">
                <div class="loader-content">
                    <div class="book-loader">
                        <div class="page-spinner"></div>
                    </div>
                    <h1 class="loader-title">Loading Book</h1>
                    <p class="loader-subtitle">Preparing your reading experience...</p>
                    <div class="loader-progress">
                        <div class="loader-progress-bar"></div>
                    </div>
                    <div class="loading-text">Please wait while we load the pages</div>
                </div>
            </div>
            
            <div class="dearflip-container" id="${containerId}" style="display: none;">
                <div class="book-content">
                    <div class="book-spine"></div>
                    <div class="dearflip-pages" id="flipbook-pages">
                        <!-- Pages will be dynamically added here -->
                    </div>
                </div>
                <button class="close-btn" onclick="goBack();" title="Back">×</button>
            </div>
        `;
    }

    /**
     * Load PDF and render all pages
     */
    async loadAndRenderPDF(container, pdfPath, options, containerId) {
        try {
            
            // Show the beautiful loader initially
            const loader = container.querySelector('#dearflip-loader');
            const bookContainer = container.querySelector(`#${containerId}`);
            
            if (loader) {
                loader.style.display = 'flex';
            }
            if (bookContainer) {
                bookContainer.style.display = 'none';
            }

            // Load PDF
            await this.pdfRenderer.loadPDF(pdfPath);
            
            // Render initial pages (first 2 pages) for immediate display
            const initialPages = await this.pdfRenderer.renderInitialPages();
            
            // Get the pages container from the book container
            const pagesContainer = bookContainer.querySelector('#flipbook-pages');
            
            // Create initial page elements
            pagesContainer.innerHTML = '';
            initialPages.forEach((page, index) => {
                const pageElement = this.createPageElement(page, index);
                pagesContainer.appendChild(pageElement);
            });
            
            // Store initial pages in instance
            const instance = this.instances.get(container.id);
            if (instance) {
                instance.totalPages = this.pdfRenderer.totalPages; // Total PDF pages
                instance.renderedPages = initialPages.length; // Currently rendered flipbook pages
                instance.pages = initialPages;
                instance.allPages = new Array(this.pdfRenderer.totalPages * 2).fill(null); // Pre-allocate array
                
                // Fill in the initial pages
                initialPages.forEach(page => {
                    instance.allPages[page.pageNumber - 1] = page;
                });
            }
            
            // Set up initial book layout - show first two pages side by side
            this.setupInitialBookLayout(bookContainer, initialPages.length);
            
            // Start loading remaining pages in background
            this.loadRemainingPagesInBackground(container, bookContainer, pagesContainer);
            
            // Smooth transition from loader to book
            setTimeout(() => {
                if (loader) {
                    loader.style.opacity = '0';
                    loader.style.transition = 'opacity 0.5s ease-out';
                }
                
                setTimeout(() => {
                    if (loader) {
                        loader.style.display = 'none';
                    }
                    if (bookContainer) {
                        bookContainer.style.display = 'block';
                        bookContainer.style.opacity = '0';
                        bookContainer.style.transition = 'opacity 0.5s ease-in';
                        
                        // Trigger reflow
                        bookContainer.offsetHeight;
                        
                        bookContainer.style.opacity = '1';
                    }
                }, 500);
            }, 1000); // Show loader for at least 1 second for better UX
            
        } catch (error) {
            console.error('Error loading PDF:', error);
            console.error('Error stack:', error.stack);
            
            // Show error in loader
            const loader = container.querySelector('#dearflip-loader');
            if (loader) {
                loader.innerHTML = `
                    <div class="loader-content">
                        <div style="font-size: 3rem; margin-bottom: 1rem;">📖</div>
                        <h1 class="loader-title" style="color: #e74c3c;">Error Loading PDF</h1>
                        <p class="loader-subtitle">${error.message}</p>
                        <button onclick="window.open('${pdfPath}', '_blank')" style="
                            background: #e74c3c; 
                            color: white; 
                            border: none; 
                            padding: 12px 24px; 
                            border-radius: 8px; 
                            cursor: pointer;
                            font-size: 1rem;
                            margin-top: 1rem;
                            transition: all 0.3s ease;
                        " onmouseover="this.style.background='#c0392b'" onmouseout="this.style.background='#e74c3c'">
                        Open PDF in New Tab
                    </button>
                </div>
            `;
            }
        }
    }

    /**
     * Set up initial book layout
     */
    setupInitialBookLayout(container, totalPages) {
        const allPages = container.querySelectorAll('.dearflip-page');
        
        // First, hide all pages
        allPages.forEach((page, index) => {
            page.style.display = 'none';
            page.style.visibility = 'hidden';
            page.style.opacity = '0';
        });
        
        // Update the instance's currentPage based on total pages
        const instance = this.instances.get(container.id);
        if (instance) {
            if (totalPages === 1) {
                // Single page document - show only page 1
                instance.currentPage = 1;
            } else {
                // Multi-page document - start showing pages 1-2
                instance.currentPage = 2;
            }
        }
        
        // Then show and position pages based on total count
        allPages.forEach((page, index) => {
            const pageNumber = parseInt(page.dataset.pageNumber);
            page.classList.remove('current-left', 'current-right', 'left-page', 'right-page');
            
            if (totalPages === 1) {
                // Single page document - show only page 1 on the left
                if (pageNumber === 1) {
                    page.classList.add('current-left');
                    page.style.display = 'block';
                    page.style.visibility = 'visible';
                    page.style.opacity = '1';
                } else {
                    page.style.display = 'none';
                    page.style.visibility = 'hidden';
                    page.style.opacity = '0';
                }
            } else {
                // Multi-page document - show pages 1-2
            if (pageNumber === 1) {
                // First page goes on the left
                page.classList.add('current-left');
                page.style.display = 'block';
                page.style.visibility = 'visible';
                page.style.opacity = '1';
            } else if (pageNumber === 2) {
                // Second page goes on the right
                page.classList.add('current-right');
                page.style.display = 'block';
                page.style.visibility = 'visible';
                page.style.opacity = '1';
            } else {
                // Other pages are hidden completely
                page.classList.add('right-page');
                page.style.display = 'none';
                page.style.visibility = 'hidden';
                page.style.opacity = '0';
                }
            }
        });
        
        
        // Update button states after initial setup
        this.updateButtonStates(container);
    }

    /**
     * Create a page element
     */
    createPageElement(page, index) {
        const pageDiv = document.createElement('div');
        pageDiv.className = 'dearflip-page';
        pageDiv.dataset.pageNumber = page.pageNumber;
        
        // Add page shadow element
        const pageShadow = document.createElement('div');
        pageShadow.className = 'page-shadow';
        pageDiv.appendChild(pageShadow);
        
        const img = document.createElement('img');
        img.src = page.dataURL;
        img.alt = `Page ${page.pageNumber}`;
        
        // Enhanced CSS for maximum clarity and visibility (optimized for fullscreen)
        const isFullscreen = !!(document.fullscreenElement || document.webkitFullscreenElement || 
                                document.mozFullScreenElement || document.msFullscreenElement);
        const contrastLevel = isFullscreen ? 1.08 : 1.05;
        const brightnessLevel = isFullscreen ? 1.03 : 1.01;
        
        img.style.cssText = `
            width: 100%;
            height: 100%;
            object-fit: contain;
            image-rendering: -webkit-optimize-contrast;
            image-rendering: crisp-edges;
            image-rendering: pixelated;
            image-rendering: auto;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
            text-rendering: optimizeLegibility;
            filter: contrast(${contrastLevel}) brightness(${brightnessLevel});
            backface-visibility: hidden;
            transform: translateZ(0);
            will-change: transform;
        `;
        
        img.onload = () => {};
        
        pageDiv.appendChild(img);
        
        // Initial positioning will be handled by setupInitialBookLayout
        
        // Add click to flip based on page position
        pageDiv.addEventListener('click', (event) => {
            // Prevent event bubbling
            event.stopPropagation();
            
            const container = pageDiv.closest('.dearflip-container');
            const pageNumber = parseInt(pageDiv.dataset.pageNumber);
            const instance = this.instances.get(container.id);
            
            if (instance) {
                // CRITICAL: Check if already flipping to prevent race conditions
                if (instance.isFlipping) {
                    console.log('Page click ignored - already flipping');
                    return;
                }
                
                // CRITICAL: Debounce rapid clicks
                const now = Date.now();
                if (instance.lastClickTime && (now - instance.lastClickTime) < 500) {
                    console.log('Page click ignored - too rapid');
                    return;
                }
                instance.lastClickTime = now;
                
                // Determine if this is the left or right page
                const isLeftPage = pageDiv.classList.contains('current-left');
                const isRightPage = pageDiv.classList.contains('current-right');
                
                console.log(`Page click detected - Left: ${isLeftPage}, Right: ${isRightPage}, Page: ${pageNumber}`);
                
                if (isRightPage) {
                    // Clicking right page = next page
                    console.log('Page click: Going to next page');
                    // Use the same method as control buttons for consistency
                    this.nextPage(container.id);
                } else if (isLeftPage) {
                    // Clicking left page = previous page
                    console.log('Page click: Going to previous page');
                    // Use the same method as control buttons for consistency
                    this.previousPage(container.id);
                }
            }
        });
        
        return pageDiv;
    }

    /**
     * Initialize 3D effects
     */
    initialize3DEffects(container, options) {
        const pages = container.querySelectorAll('.dearflip-page');
        
        pages.forEach((page, index) => {
            // Add 3D positioning
            page.style.transform = `translateZ(${index * 2}px)`;
            
            // Add hover effects - only translateZ, no scaling to keep center binding fixed
            page.addEventListener('mouseenter', () => {
                if (page.style.display !== 'none') {
                    page.style.transform = `translateZ(${index * 2 + 20}px)`;
                }
            });
            
            page.addEventListener('mouseleave', () => {
                if (page.style.display !== 'none') {
                    page.style.transform = `translateZ(${index * 2}px)`;
                }
            });
        });
        
        // Initialize dragging functionality on the main PDF viewer container
        const pdfViewerContainer = document.querySelector('.pdf-viewer-container');
        if (pdfViewerContainer) {
            this.initializeDragging(pdfViewerContainer);
        }
    }
    
    /**
     * Initialize dragging functionality for zoomed content
     */
    initializeDragging(pdfViewerContainer) {
        let isDragging = false;
        let startX = 0;
        let startY = 0;
        let currentX = 0;
        let currentY = 0;
        let isZoomed = false;
        
        // Get the actual book container that will be transformed
        const getBookContainer = () => {
            return document.querySelector('.dearflip-container');
        };
        
        // Check if content is zoomed (scale > 1)
        const checkZoom = () => {
            const bookContainer = getBookContainer();
            if (!bookContainer) return;
            
            const transform = bookContainer.style.transform || '';
            const scaleMatch = transform.match(/scale\(([^)]+)\)/);
            const scale = scaleMatch ? parseFloat(scaleMatch[1]) : 1;
            isZoomed = scale > 1;
            
            if (isZoomed) {
                pdfViewerContainer.classList.add('draggable');
                pdfViewerContainer.style.cursor = 'grab';
            } else {
                pdfViewerContainer.classList.remove('draggable');
                pdfViewerContainer.classList.remove('dragging');
                pdfViewerContainer.style.cursor = '';
                // Reset position when not zoomed
                const scaleValue = scaleMatch ? scaleMatch[1] : '1';
                bookContainer.style.transform = `scale(${scaleValue})`;
                currentX = 0;
                currentY = 0;
            }
        };
        
        // Mouse events
        pdfViewerContainer.addEventListener('mousedown', (e) => {
            if (!isZoomed) return;
            
            isDragging = true;
            pdfViewerContainer.classList.add('dragging');
            pdfViewerContainer.style.cursor = 'grabbing';
            startX = e.clientX - currentX;
            startY = e.clientY - currentY;
            e.preventDefault();
        });

        // Add hover effect for better UX
        pdfViewerContainer.addEventListener('mouseenter', () => {
            if (isZoomed) {
                pdfViewerContainer.style.cursor = 'grab';
            }
        });

        pdfViewerContainer.addEventListener('mouseleave', () => {
            if (!isDragging) {
                pdfViewerContainer.style.cursor = '';
            }
        });
        
        document.addEventListener('mousemove', (e) => {
            if (!isDragging || !isZoomed) return;
            
            currentX = e.clientX - startX;
            currentY = e.clientY - startY;
            
            // Update transform with translate while preserving scale on the book container
            const bookContainer = getBookContainer();
            if (!bookContainer) return;
            
            const transform = bookContainer.style.transform || '';
            const scaleMatch = transform.match(/scale\(([^)]+)\)/);
            const scaleValue = scaleMatch ? scaleMatch[1] : '1';
            
            // Remove existing translate and add new one
            const newTransform = transform.replace(/translate\([^)]*\)/g, '') + ` translate(${currentX}px, ${currentY}px)`;
            bookContainer.style.transform = newTransform;
        });
        
        document.addEventListener('mouseup', () => {
            if (isDragging) {
                isDragging = false;
                pdfViewerContainer.classList.remove('dragging');
                // Reset cursor to grab when not dragging
                if (isZoomed) {
                    pdfViewerContainer.style.cursor = 'grab';
                }
            }
        });
        
        // Touch events for mobile
        pdfViewerContainer.addEventListener('touchstart', (e) => {
            if (!isZoomed) return;
            
            isDragging = true;
            pdfViewerContainer.classList.add('dragging');
            const touch = e.touches[0];
            startX = touch.clientX - currentX;
            startY = touch.clientY - currentY;
            e.preventDefault();
        });
        
        document.addEventListener('touchmove', (e) => {
            if (!isDragging || !isZoomed) return;
            
            const touch = e.touches[0];
            currentX = touch.clientX - startX;
            currentY = touch.clientY - startY;
            
            // Update transform with translate while preserving scale on the book container
            const bookContainer = getBookContainer();
            if (!bookContainer) return;
            
            const transform = bookContainer.style.transform || '';
            const scaleMatch = transform.match(/scale\(([^)]+)\)/);
            const scaleValue = scaleMatch ? scaleMatch[1] : '1';
            
            // Remove existing translate and add new one
            const newTransform = transform.replace(/translate\([^)]*\)/g, '') + ` translate(${currentX}px, ${currentY}px)`;
            bookContainer.style.transform = newTransform;
            e.preventDefault();
        });
        
        document.addEventListener('touchend', () => {
            if (isDragging) {
                isDragging = false;
                pdfViewerContainer.classList.remove('dragging');
            }
        });
        
        // Monitor zoom changes on the book container
        const observer = new MutationObserver(checkZoom);
        const bookContainer = getBookContainer();
        if (bookContainer) {
            observer.observe(bookContainer, { 
                attributes: true, 
                attributeFilter: ['style'] 
            });
        }
        
        // Initial check
        checkZoom();
    }

    /**
     * Load remaining pages in background with progressive loading
     */
    async loadRemainingPagesInBackground(container, bookContainer, pagesContainer) {
        const instance = this.instances.get(container.id);
        if (!instance) return;

        // Calculate starting page for background loading
        const startPage = Math.ceil(instance.renderedPages / 2) + 1;
        
        // Start loading remaining pages
        this.pdfRenderer.renderRemainingPages(startPage, (pageData) => {
            // This callback is called for each page as it becomes ready
            this.addPageToBook(container, bookContainer, pagesContainer, pageData);
        }).then(() => {
            console.log('All remaining pages loaded in background');
        }).catch((error) => {
            console.error('Error loading remaining pages:', error);
        });
    }

    /**
     * Predictive page preloading with rolling buffer
     */
    async predictivePreload(containerId) {
        const instance = this.instances.get(containerId);
        if (!instance) return;

        // Create rolling buffer of ±2 spreads
        const bufferPages = [
            instance.currentPage + 3,
            instance.currentPage + 4,
            instance.currentPage - 3,
            instance.currentPage - 4
        ].filter(p => p > 0 && p <= instance.totalPages);

        if (bufferPages.length > 0) {
            console.log('Predictive preloading pages:', bufferPages);
            this.pdfRenderer.preloadPages(bufferPages);
        }
    }

    /**
     * Enhanced page loading with Image.decode() for GPU readiness
     */
    async loadPageWithDecode(pageData) {
        return new Promise((resolve, reject) => {
            const img = new Image();
            img.onload = async () => {
                try {
                    // Ensure image is GPU-ready
                    if (img.decode) {
                        await img.decode();
                    }
                    resolve({
                        ...pageData,
                        dataURL: img.src,
                        isGPUReady: true
                    });
                } catch (error) {
                    console.warn('Image decode failed, continuing anyway:', error);
                    resolve({
                        ...pageData,
                        dataURL: img.src,
                        isGPUReady: false
                    });
                }
            };
            img.onerror = () => reject(new Error(`Failed to load page ${pageData.pageNumber}`));
            img.src = pageData.dataURL;
        });
    }

    /**
     * Add a single page to the book as it becomes ready
     */
    addPageToBook(container, bookContainer, pagesContainer, pageData) {
        const instance = this.instances.get(container.id);
        if (!instance) return;

        // Store the page data
        instance.allPages[pageData.pageNumber - 1] = pageData;
        instance.renderedPages++;

        // Create and add the page element
        const pageElement = this.createPageElement(pageData, pageData.pageNumber - 1);
        pagesContainer.appendChild(pageElement);
        
        // Trigger preloading of adjacent pages
        this.pdfRenderer.preloadPages([pageData.pageNumber]);

        // Initially hide the page (it will be shown when needed)
        pageElement.style.display = 'none';
        pageElement.style.visibility = 'hidden';
        pageElement.style.opacity = '0';

        // CRITICAL: Ensure the image is properly loaded
        const img = pageElement.querySelector('img');
        if (img) {
            img.onload = () => {
                console.log(`Page ${pageData.pageNumber} image fully loaded`);
            };
            img.onerror = () => {
                console.error(`Page ${pageData.pageNumber} image failed to load`);
            };
        }

        console.log(`Page ${pageData.pageNumber} loaded and added to book`);
    }

    /**
     * Load specific pages on demand with aggressive preloading
     */
    async loadPagesOnDemand(container, pageNumbers) {
        const instance = this.instances.get(container.id);
        if (!instance) return;

        const pagesContainer = container.querySelector('#flipbook-pages');
        const bookContainer = container.querySelector(`#${container.id}`);

        // CRITICAL: Load pages with verification
        const loadPromises = pageNumbers.map(async (pageNumber) => {
            // Check if page is already loaded
            if (instance.allPages[pageNumber - 1]) {
                console.log(`Page ${pageNumber} already loaded`);
                return true;
            }

            try {
                console.log(`Loading page ${pageNumber} at low quality for fast display...`);
                // Step 4: Get the page data from PDF renderer at LOW quality for fast display
                const pageData = await this.pdfRenderer.renderPageLowQuality(pageNumber);
                
                // Add to book
                this.addPageToBook(container, bookContainer, pagesContainer, pageData);
                
                // Trigger preloading of adjacent pages
                this.pdfRenderer.preloadPages([pageNumber]);
                
                // CRITICAL: Wait for DOM to be updated
                await new Promise(resolve => setTimeout(resolve, 100));
                
                // CRITICAL: Verify page is actually in DOM
                const pageElement = container.querySelector(`[data-page-number="${pageNumber}"]`);
                if (!pageElement) {
                    throw new Error(`Page ${pageNumber} not found in DOM after loading`);
                }
                
                // CRITICAL: Verify image is loaded
                const img = pageElement.querySelector('img');
                if (img && (!img.complete || img.naturalWidth === 0)) {
                    console.log(`Waiting for page ${pageNumber} image to load...`);
                    await new Promise((resolve, reject) => {
                        const timeout = setTimeout(() => {
                            reject(new Error(`Page ${pageNumber} image load timeout`));
                        }, 5000);
                        
                        img.onload = () => {
                            clearTimeout(timeout);
                            resolve();
                        };
                        img.onerror = () => {
                            clearTimeout(timeout);
                            reject(new Error(`Page ${pageNumber} image failed to load`));
                        };
                    });
                }
                
                console.log(`Page ${pageNumber} loaded and verified successfully`);
                return true;
            } catch (error) {
                console.error(`Error loading page ${pageNumber} on demand:`, error);
                return false;
            }
        });

        // Wait for all pages to load
        const results = await Promise.all(loadPromises);
        const allLoaded = results.every(result => result === true);
        
        if (!allLoaded) {
            throw new Error('Some pages failed to load');
        }
        
        console.log('All pages loaded successfully');
    }

    /**
     * Verify page readiness before flipping
     */
    async verifyPageReadiness(container, pageNumbers) {
        const instance = this.instances.get(container.id);
        if (!instance) return false;

        for (const pageNumber of pageNumbers) {
            // Check if page data exists
            if (!instance.allPages[pageNumber - 1]) {
                console.log(`Page ${pageNumber} data not found`);
                return false;
            }

            // Check if page element exists in DOM
            const pageElement = container.querySelector(`[data-page-number="${pageNumber}"]`);
            if (!pageElement) {
                console.log(`Page ${pageNumber} element not found in DOM`);
                return false;
            }

            // Check if image is loaded
            const img = pageElement.querySelector('img');
            if (!img) {
                console.log(`Page ${pageNumber} image element not found`);
                return false;
            }

            if (!img.complete || img.naturalWidth === 0) {
                console.log(`Page ${pageNumber} image not loaded`);
                return false;
            }

            // Check if image is visible
            if (img.style.display === 'none' || img.style.visibility === 'hidden') {
                console.log(`Page ${pageNumber} image not visible`);
                return false;
            }
        }

        console.log('All pages verified as ready');
        return true;
    }

    /**
     * Flip to next page with optimized rendering
     */
    async flipToNextPage(container) {
        if (!container) {
            return;
        }
        
        const instance = this.instances.get(container.id);
        if (!instance) {
            return;
        }
        
        // Prevent multiple simultaneous flips
        if (instance.isFlipping) {
            console.log('Page flip already in progress, ignoring request');
            return;
        }
        
        // Cancel any existing flip
        if (instance.flipController) {
            instance.flipController.abort();
        }
        
        instance.flipController = new AbortController();
        instance.isFlipping = true;
        instance.flipStartTime = Date.now();
        
        if (instance.totalPages === 1) {
            console.log('Cannot flip to next page - single page document');
            instance.isFlipping = false;
            return;
        }
        
        if (instance.currentPage >= instance.totalPages) {
            console.log('Cannot flip to next page - already at last page');
            instance.isFlipping = false;
            return;
        }

        const currentRightPage = container.querySelector('.current-right');
        const nextLeftPageNumber = instance.currentPage + 1;  // The page that will become the new left page
        const nextRightPageNumber = instance.currentPage + 2; // The page that will become the new right page
        const nextLeftPageElement = container.querySelector(`[data-page-number="${nextLeftPageNumber}"]`);
        const nextRightPageElement = container.querySelector(`[data-page-number="${nextRightPageNumber}"]`);
        
        // CRITICAL: Use new verification system
        this.verifyPageReadiness(container, [nextLeftPageNumber, nextRightPageNumber]).then(async (isReady) => {
            if (!isReady) {
                console.log('Pages not ready, loading on demand...');
                
                // Show loading indicator
                if (typeof window.showPageLoadingIndicator === 'function') {
                    window.showPageLoadingIndicator();
                }
                
                try {
                    await this.loadPagesOnDemand(container, [nextLeftPageNumber, nextRightPageNumber]);
                    
                    // Hide loading indicator
                    if (typeof window.hidePageLoadingIndicator === 'function') {
                        window.hidePageLoadingIndicator();
                    }
                    
                    // CRITICAL: Verify again after loading
                    const isReadyAfterLoad = await this.verifyPageReadiness(container, [nextLeftPageNumber, nextRightPageNumber]);
                    if (isReadyAfterLoad) {
                        console.log('Pages loaded and verified, retrying flip...');
                        // Reset flipping flag before retry
                        instance.isFlipping = false;
                        // Retry the flip after pages are loaded
                        this.flipToNextPage(container);
                    } else {
                        console.error('CRITICAL: Pages still not ready after loading!');
                        instance.isFlipping = false;
                    }
                } catch (error) {
                    // Hide loading indicator on error
                    if (typeof window.hidePageLoadingIndicator === 'function') {
                        window.hidePageLoadingIndicator();
                    }
                    console.error('Error loading pages on demand:', error);
                    // Reset flipping flag on error
                    instance.isFlipping = false;
                }
                return;
            }
            
            // Pages are ready, continue with flip
            console.log('Pages verified as ready, proceeding with flip...');
            this.proceedWithNextPageFlip(container, instance, currentRightPage, nextLeftPageElement, nextRightPageElement, nextLeftPageNumber, nextRightPageNumber);
        });
    }

    /**
     * Proceed with next page flip after verification with optimized rendering
     */
    async proceedWithNextPageFlip(container, instance, currentRightPage, nextLeftPageElement, nextRightPageElement, nextLeftPageNumber, nextRightPageNumber) {
        
        try {
            // Ensure pages are pre-decoded and GPU-ready
            await Promise.all([
                this.ensurePageReady(nextLeftPageElement),
                this.ensurePageReady(nextRightPageElement)
            ]);
            
            // Check if flip was cancelled
            if (instance.flipController?.signal.aborted) {
                console.log('Flip cancelled, aborting');
                instance.isFlipping = false;
                return;
            }
        } catch (error) {
            console.warn('Page readiness check failed, continuing anyway:', error);
        }
        
        if (currentRightPage && nextLeftPageElement && nextRightPageElement) {
            // CRITICAL: Verify pages are actually in the DOM
            if (!document.contains(nextLeftPageElement) || !document.contains(nextRightPageElement)) {
                console.error('CRITICAL: Page elements not in DOM! Retrying flip...');
                instance.isFlipping = false;
                setTimeout(() => this.flipToNextPage(container), 100);
                return;
            }
            
            // Prepare the next pages to be visible behind the flipping page
            nextLeftPageElement.style.display = 'block';
            nextLeftPageElement.style.visibility = 'visible';
            nextLeftPageElement.style.opacity = '0.95'; // Start with high opacity
            nextLeftPageElement.style.zIndex = '4';
            nextLeftPageElement.style.pointerEvents = 'none'; // Prevent interaction during flip
            nextLeftPageElement.style.backfaceVisibility = 'visible';
            nextLeftPageElement.style.webkitBackfaceVisibility = 'visible';
            
            nextRightPageElement.style.display = 'block';
            nextRightPageElement.style.visibility = 'visible';
            nextRightPageElement.style.opacity = '0.98'; // Start with very high opacity
            nextRightPageElement.style.zIndex = '5';
            nextRightPageElement.style.pointerEvents = 'none'; // Prevent interaction during flip
            nextRightPageElement.style.backfaceVisibility = 'visible';
            nextRightPageElement.style.webkitBackfaceVisibility = 'visible';
            
            // Ensure the flipping page stays visible
            currentRightPage.style.display = 'block';
            currentRightPage.style.visibility = 'visible';
            currentRightPage.style.opacity = '1';
            currentRightPage.style.pointerEvents = 'none'; // Prevent interaction during flip
            currentRightPage.style.backfaceVisibility = 'visible';
            currentRightPage.style.webkitBackfaceVisibility = 'visible';
            
            // Create a backface for the flipping page with the correct content
            const backfaceContent = nextLeftPageElement.cloneNode(true);
            backfaceContent.style.position = 'absolute';
            backfaceContent.style.top = '0';
            backfaceContent.style.left = '0';
            backfaceContent.style.width = '100%';
            backfaceContent.style.height = '100%';
            backfaceContent.style.transform = 'rotateY(180deg)';
            backfaceContent.style.backfaceVisibility = 'visible';
            backfaceContent.style.webkitBackfaceVisibility = 'visible';
            backfaceContent.style.zIndex = '1';
            backfaceContent.style.pointerEvents = 'none';
            backfaceContent.style.opacity = '0'; // Start invisible
            backfaceContent.style.visibility = 'hidden'; // Start hidden
            backfaceContent.style.display = 'block';
            
            // Ensure the backface content is properly styled
            const backfaceImages = backfaceContent.querySelectorAll('img');
            backfaceImages.forEach(img => {
                img.style.opacity = '0'; // Start invisible
                img.style.visibility = 'hidden'; // Start hidden
                img.style.display = 'block';
                img.style.backfaceVisibility = 'visible';
                img.style.webkitBackfaceVisibility = 'visible';
            });
            
            currentRightPage.appendChild(backfaceContent);
            
            // Show backface content only when flip animation starts
            setTimeout(() => {
                backfaceContent.style.opacity = '1';
                backfaceContent.style.visibility = 'visible';
                backfaceImages.forEach(img => {
                    img.style.opacity = '1';
                    img.style.visibility = 'visible';
                });
            }, 150); // Slightly longer delay for more seamless transition
            
            // Single page flip: Only flip the right page
            currentRightPage.classList.add('page-flip', 'flipping');
            console.log('Single page flip: Flipping right page forward');
            
            // Simplified: Remove complex shadow animation to prevent freezing
            // this.animateFlipShadows(currentRightPage, 'forward');
            
            // Simplified: Remove audio to prevent blocking
            // setTimeout(() => {
            //     this.playPageTurnSound();
            // }, 600);
            
            // Simplified: Basic reveal without complex animation
            nextLeftPageElement.style.opacity = '1';
            nextRightPageElement.style.opacity = '1';
            
            // Move pages after animation
            // Use animation event listener instead of setTimeout for more reliable completion
            let animationCompleted = false;
            
            const handleAnimationEnd = () => {
                if (animationCompleted) {
                    console.log('Animation already completed, ignoring duplicate event');
                    return;
                }
                
                animationCompleted = true;
                console.log('Animation ended, finalizing page positions...');
                
                // Clean up animation classes and clear transform
                currentRightPage.classList.remove('page-flip', 'flipping');
                currentRightPage.style.transform = '';
                
                // First, get the current left page BEFORE we change classes
                const currentLeftPage = container.querySelector('.current-left');
                console.log('Current left page before changes:', currentLeftPage);
                
                // Move current left page to background (hidden) first using deferred updates
                if (currentLeftPage && currentLeftPage !== currentRightPage) {
                    currentLeftPage.classList.remove('current-left');
                    currentLeftPage.classList.add('left-page');
                    this.deferVisibilityUpdate(currentLeftPage, 'hide');
                    currentLeftPage.style.zIndex = '1';
                    console.log('Moved current left page to background');
                }
                
                // Set up the new left page (nextLeftPageElement) using batch writes
                nextLeftPageElement.classList.remove('right-page');
                nextLeftPageElement.classList.add('current-left');
                this.batchStyleWrites(nextLeftPageElement, {
                    display: 'block',
                    visibility: 'visible',
                    opacity: '1',
                    zIndex: '5',
                    pointerEvents: 'auto',
                    backfaceVisibility: 'visible',
                    webkitBackfaceVisibility: 'visible',
                    transform: ''
                });
                
                // Ensure the left page image is properly loaded and visible
                const leftPageImg = nextLeftPageElement.querySelector('img');
                if (leftPageImg) {
                    leftPageImg.style.display = 'block';
                    leftPageImg.style.visibility = 'visible';
                    leftPageImg.style.opacity = '1';
                    leftPageImg.style.backfaceVisibility = 'visible';
                    leftPageImg.style.webkitBackfaceVisibility = 'visible';
                    
                    // Force image reload if it's not loaded
                    if (!leftPageImg.complete || leftPageImg.naturalWidth === 0) {
                        console.log('Left page image not loaded, forcing reload...');
                        const originalSrc = leftPageImg.src;
                        leftPageImg.src = '';
                        leftPageImg.src = originalSrc;
                    }
                }
                
                console.log('Set up new left page:', nextLeftPageNumber);
                
                // Set up the new right page (nextRightPageElement) using batch writes
                nextRightPageElement.classList.remove('right-page');
                nextRightPageElement.classList.add('current-right');
                this.batchStyleWrites(nextRightPageElement, {
                    display: 'block',
                    visibility: 'visible',
                    opacity: '1',
                    zIndex: '5',
                    pointerEvents: 'auto',
                    backfaceVisibility: 'visible',
                    webkitBackfaceVisibility: 'visible',
                    transform: ''
                });
                
                // CRITICAL: Enforce right page visibility immediately
                if (typeof window.enforceRightPageVisibility === 'function') {
                    window.enforceRightPageVisibility(container.id);
                }
                
                // AGGRESSIVE FIX: Ensure the right page image is properly loaded and visible
                const rightPageImg = nextRightPageElement.querySelector('img');
                if (rightPageImg) {
                    rightPageImg.style.display = 'block';
                    rightPageImg.style.visibility = 'visible';
                    rightPageImg.style.opacity = '1';
                    rightPageImg.style.backfaceVisibility = 'visible';
                    rightPageImg.style.webkitBackfaceVisibility = 'visible';
                    
                    // Multiple fallback mechanisms for image loading
                    if (!rightPageImg.complete || rightPageImg.naturalWidth === 0) {
                        console.log('CRITICAL: Right page image not loaded, applying aggressive fixes...');
                        
                        // Method 1: Force reload with cache busting
                        const originalSrc = rightPageImg.src;
                        rightPageImg.src = '';
                        rightPageImg.src = originalSrc + '?t=' + Date.now();
                        
                        // Method 2: Set up multiple onload handlers
                        rightPageImg.onload = () => {
                            console.log('Right page image loaded via onload handler');
                            rightPageImg.style.display = 'block';
                            rightPageImg.style.visibility = 'visible';
                            rightPageImg.style.opacity = '1';
                        };
                        
                        rightPageImg.onerror = () => {
                            console.log('Right page image failed to load, trying alternative...');
                            // Method 3: Try without cache busting
                            rightPageImg.src = originalSrc;
                        };
                        
                        // Method 4: Force visibility after a delay
                        setTimeout(() => {
                            if (!rightPageImg.complete || rightPageImg.naturalWidth === 0) {
                                console.log('CRITICAL: Forcing right page visibility despite loading issues');
                                rightPageImg.style.display = 'block';
                                rightPageImg.style.visibility = 'visible';
                                rightPageImg.style.opacity = '1';
                                rightPageImg.style.background = 'white';
                            }
                        }, 500);
                    }
                }
                
                // CRITICAL: Force right page visibility regardless of image status
                nextRightPageElement.style.display = 'block';
                nextRightPageElement.style.visibility = 'visible';
                nextRightPageElement.style.opacity = '1';
                nextRightPageElement.style.zIndex = '5';
                nextRightPageElement.style.pointerEvents = 'auto';
                
                console.log('Set up new right page:', nextRightPageNumber);
                
                // CRITICAL SAFETY: Multiple checks to ensure right page is visible
                setTimeout(() => {
                    const allPages = container.querySelectorAll('.dearflip-page');
                    allPages.forEach(page => {
                        if (page.classList.contains('current-left') || page.classList.contains('current-right')) {
                            page.style.pointerEvents = 'auto';
                        }
                    });
                    console.log('Ensured all current pages have pointer events enabled');
                    
                    // CRITICAL: Double-check right page visibility
                    const currentRightPage = container.querySelector('.current-right');
                    if (currentRightPage) {
                        console.log('CRITICAL CHECK: Verifying right page visibility...');
                        currentRightPage.style.display = 'block';
                        currentRightPage.style.visibility = 'visible';
                        currentRightPage.style.opacity = '1';
                        currentRightPage.style.zIndex = '5';
                        
                        const rightImg = currentRightPage.querySelector('img');
                        if (rightImg) {
                            rightImg.style.display = 'block';
                            rightImg.style.visibility = 'visible';
                            rightImg.style.opacity = '1';
                            
                            // If image still not loaded, force a white background
                            if (!rightImg.complete || rightImg.naturalWidth === 0) {
                                console.log('CRITICAL: Right page image still not loaded, applying emergency fix');
                                rightImg.style.background = 'white';
                                rightImg.style.minHeight = '100%';
                                rightImg.style.minWidth = '100%';
                            }
                        }
                    }
                }, 100);
                
                // ADDITIONAL SAFETY: Another check after 1 second
                setTimeout(() => {
                    const currentRightPage = container.querySelector('.current-right');
                    if (currentRightPage) {
                        const rightImg = currentRightPage.querySelector('img');
                        if (rightImg && (!rightImg.complete || rightImg.naturalWidth === 0)) {
                            console.log('EMERGENCY: Right page still not loaded after 1 second, forcing visibility');
                            currentRightPage.style.display = 'block';
                            currentRightPage.style.visibility = 'visible';
                            currentRightPage.style.opacity = '1';
                            rightImg.style.display = 'block';
                            rightImg.style.visibility = 'visible';
                            rightImg.style.opacity = '1';
                            rightImg.style.background = 'white';
                        }
                    }
                }, 1000);
                
                // Hide the old current right page (it was flipping)
                currentRightPage.classList.remove('current-right', 'page-flip', 'flipping');
                currentRightPage.classList.add('right-page');
                currentRightPage.style.display = 'none';
                currentRightPage.style.visibility = 'hidden';
                currentRightPage.style.opacity = '0';
                currentRightPage.style.zIndex = '1';
                currentRightPage.style.transform = '';
                
                // Remove the backface content that was added during the flip
                const backfaceContent = currentRightPage.querySelector('[style*="rotateY(180deg)"]');
                if (backfaceContent) {
                    backfaceContent.remove();
                }
                
                console.log('Hidden old current right page and cleaned up backface');
                
                instance.currentPage += 2; // Increment by 2 since we show 2 pages at a time
                
                // Update button states after page flip
                this.updateButtonStates(container);
                
                // Trigger page info update if callback exists
                if (typeof window.updatePageInfo === 'function') {
                    setTimeout(() => window.updatePageInfo(), 100);
                }
                
                console.log(`Page flip complete. Now showing pages ${instance.currentPage - 1} and ${instance.currentPage}`);
                
                // Reset flipping flag and timing
                instance.currentPage = nextLeftPageNumber + 1;
                instance.isFlipping = false;
                instance.flipStartTime = null;
                instance.flipController = null;
                
                // Step 5: Silently upgrade visible pages to high quality after flip completes
                // Run in background without blocking or triggering events
                setTimeout(() => {
                    this.upgradeVisiblePagesQuality(container, [nextLeftPageNumber, nextRightPageNumber]);
                }, 100);
                
                // Trigger predictive preloading for next pages
                this.predictivePreload(container.id);
                
                // Remove the event listener to prevent memory leaks
                currentRightPage.removeEventListener('animationend', handleAnimationEnd);
                currentRightPage.removeEventListener('webkitAnimationEnd', handleAnimationEnd);
            };
            
            // Add event listeners for animation end
            currentRightPage.addEventListener('animationend', handleAnimationEnd);
            currentRightPage.addEventListener('webkitAnimationEnd', handleAnimationEnd);
            
            // Enhanced fallback timeout with multiple checks
            setTimeout(() => {
                if (!animationCompleted) {
                    console.log('Fallback timeout reached, ensuring animation completion...');
                    if (currentRightPage.classList.contains('page-flip')) {
                        handleAnimationEnd();
                    }
                } else {
                    console.log('Animation already completed, skipping fallback timeout');
                }
            }, 1500); // Slightly longer than animation duration
            
            // Additional safety check - force reset if stuck
            setTimeout(() => {
                if (instance.isFlipping) {
                    console.warn('CRITICAL: Flip appears stuck, forcing reset...');
                    instance.isFlipping = false;
                    // Clean up any stuck animation classes
                    currentRightPage.classList.remove('page-flip', 'flipping');
                    currentRightPage.style.transform = '';
                }
            }, 3000); // Emergency reset after 3 seconds
        } else {
            console.error('Cannot flip page - missing elements:');
            console.error('currentRightPage:', currentRightPage);
            console.error('nextPageElement:', nextPageElement);
        }
    }


    /**
     * Flip to previous page with optimized rendering
     */
    async flipToPreviousPage(container) {
        console.log('flipToPreviousPage called with container:', container);
        if (!container) {
            console.error('No container provided to flipToPreviousPage');
            return;
        }
        
        const instance = this.instances.get(container.id);
        console.log('Instance found:', instance);
        if (!instance) {
            console.error('No instance found for container:', container.id);
            return;
        }
        
        // Prevent multiple simultaneous flips
        if (instance.isFlipping) {
            console.log('Page flip already in progress, ignoring previous page request');
            return;
        }
        
        // Cancel any existing flip
        if (instance.flipController) {
            instance.flipController.abort();
        }
        
        instance.flipController = new AbortController();
        instance.isFlipping = true;
        instance.flipStartTime = Date.now();
        
        if (instance.totalPages === 1) {
            console.log('Cannot flip to previous page - single page document');
            instance.isFlipping = false;
            return;
        }
        
        if (instance.currentPage <= 2) {
            console.log('Cannot flip to previous page - already at first page');
            instance.isFlipping = false;
            return;
        }

        const currentLeftPage = container.querySelector('.current-left');
        const prevLeftPageNumber = instance.currentPage - 3;  // The page that will become the new left page
        const prevRightPageNumber = instance.currentPage - 2; // The page that will become the new right page
        const prevLeftPageElement = container.querySelector(`[data-page-number="${prevLeftPageNumber}"]`);
        const prevRightPageElement = container.querySelector(`[data-page-number="${prevRightPageNumber}"]`);
        
        // Check if pages are loaded, if not, load them on demand
        if (!prevLeftPageElement || !prevRightPageElement) {
            console.log('Previous pages not loaded yet, loading on demand...');
            
            // Show loading indicator
            if (typeof window.showPageLoadingIndicator === 'function') {
                window.showPageLoadingIndicator();
            }
            
            this.loadPagesOnDemand(container, [prevLeftPageNumber, prevRightPageNumber]).then(() => {
                // Hide loading indicator
                if (typeof window.hidePageLoadingIndicator === 'function') {
                    window.hidePageLoadingIndicator();
                }
                
                // CRITICAL: Wait for DOM to be updated and images to be ready
                setTimeout(() => {
                    // Double-check that pages are now available
                    const newPrevLeftPageElement = container.querySelector(`[data-page-number="${prevLeftPageNumber}"]`);
                    const newPrevRightPageElement = container.querySelector(`[data-page-number="${prevRightPageNumber}"]`);
                    
                    if (newPrevLeftPageElement && newPrevRightPageElement) {
                        console.log('Previous pages loaded successfully, retrying flip...');
                        // Reset flipping flag before retry
                        instance.isFlipping = false;
                        // Retry the flip after pages are loaded
                        this.flipToPreviousPage(container);
                    } else {
                        console.error('CRITICAL: Previous pages still not found after loading!');
                        instance.isFlipping = false;
                    }
                }, 200); // Wait 200ms for DOM updates
            }).catch((error) => {
                // Hide loading indicator on error
                if (typeof window.hidePageLoadingIndicator === 'function') {
                    window.hidePageLoadingIndicator();
                }
                console.error('Error loading pages on demand:', error);
                // Reset flipping flag on error
                instance.isFlipping = false;
            });
            return;
        }
        
        // Additional check: Ensure page images are loaded before flipping
        const prevLeftImg = prevLeftPageElement.querySelector('img');
        const prevRightImg = prevRightPageElement.querySelector('img');
        
        if (prevLeftImg && (!prevLeftImg.complete || prevLeftImg.naturalWidth === 0)) {
            console.log('Previous left page image not loaded, waiting...');
            prevLeftImg.onload = () => {
                console.log('Previous left page image loaded, retrying flip...');
                instance.isFlipping = false;
                setTimeout(() => this.flipToPreviousPage(container), 100);
            };
            return;
        }
        
        if (prevRightImg && (!prevRightImg.complete || prevRightImg.naturalWidth === 0)) {
            console.log('Previous right page image not loaded, waiting...');
            prevRightImg.onload = () => {
                console.log('Previous right page image loaded, retrying flip...');
                instance.isFlipping = false;
                setTimeout(() => this.flipToPreviousPage(container), 100);
            };
            return;
        }
        
        if (currentLeftPage && prevLeftPageElement && prevRightPageElement) {
            // Prepare the previous pages to be visible behind the flipping page
            prevLeftPageElement.style.display = 'block';
            prevLeftPageElement.style.visibility = 'visible';
            prevLeftPageElement.style.opacity = '0.95'; // Start with high opacity
            prevLeftPageElement.style.zIndex = '4';
            prevLeftPageElement.style.pointerEvents = 'none'; // Prevent interaction during flip
            prevLeftPageElement.style.backfaceVisibility = 'visible';
            prevLeftPageElement.style.webkitBackfaceVisibility = 'visible';
            
            prevRightPageElement.style.display = 'block';
            prevRightPageElement.style.visibility = 'visible';
            prevRightPageElement.style.opacity = '0.98'; // Start with very high opacity
            prevRightPageElement.style.zIndex = '5';
            prevRightPageElement.style.pointerEvents = 'none'; // Prevent interaction during flip
            prevRightPageElement.style.backfaceVisibility = 'visible';
            prevRightPageElement.style.webkitBackfaceVisibility = 'visible';
            
            // Ensure the flipping page stays visible
            currentLeftPage.style.display = 'block';
            currentLeftPage.style.visibility = 'visible';
            currentLeftPage.style.opacity = '1';
            currentLeftPage.style.pointerEvents = 'none'; // Prevent interaction during flip
            currentLeftPage.style.backfaceVisibility = 'visible';
            currentLeftPage.style.webkitBackfaceVisibility = 'visible';
            
            // Create a backface for the flipping page with the correct content
            const backfaceContent = prevRightPageElement.cloneNode(true);
            backfaceContent.style.position = 'absolute';
            backfaceContent.style.top = '0';
            backfaceContent.style.left = '0';
            backfaceContent.style.width = '100%';
            backfaceContent.style.height = '100%';
            backfaceContent.style.transform = 'rotateY(180deg)';
            backfaceContent.style.backfaceVisibility = 'visible';
            backfaceContent.style.webkitBackfaceVisibility = 'visible';
            backfaceContent.style.zIndex = '1';
            backfaceContent.style.pointerEvents = 'none';
            backfaceContent.style.opacity = '0'; // Start invisible
            backfaceContent.style.visibility = 'hidden'; // Start hidden
            backfaceContent.style.display = 'block';
            
            // Ensure the backface content is properly styled
            const backfaceImages = backfaceContent.querySelectorAll('img');
            backfaceImages.forEach(img => {
                img.style.opacity = '0'; // Start invisible
                img.style.visibility = 'hidden'; // Start hidden
                img.style.display = 'block';
                img.style.backfaceVisibility = 'visible';
                img.style.webkitBackfaceVisibility = 'visible';
            });
            
            currentLeftPage.appendChild(backfaceContent);
            
            // Show backface content only when flip animation starts
            setTimeout(() => {
                backfaceContent.style.opacity = '1';
                backfaceContent.style.visibility = 'visible';
                backfaceImages.forEach(img => {
                    img.style.opacity = '1';
                    img.style.visibility = 'visible';
                });
            }, 150); // Slightly longer delay for more seamless transition
            
            // Single page flip: Only flip the left page backward
            currentLeftPage.classList.add('page-flip', 'reverse', 'flipping');
            console.log('Single page flip: Flipping left page backward');
            
            // Simplified: Remove complex shadow animation to prevent freezing
            // this.animateFlipShadows(currentLeftPage, 'backward');
            
            // Simplified: Remove audio to prevent blocking
            // setTimeout(() => {
            //     this.playPageTurnSound();
            // }, 600);
            
            // Simplified: Basic reveal without complex animation
            prevLeftPageElement.style.opacity = '1';
            prevRightPageElement.style.opacity = '1';
            
            // Move pages after animation
            // Use animation event listener instead of setTimeout for more reliable completion
            let previousAnimationCompleted = false;
            
            const handlePreviousAnimationEnd = () => {
                if (previousAnimationCompleted) {
                    console.log('Previous animation already completed, ignoring duplicate event');
                    return;
                }
                
                previousAnimationCompleted = true;
                console.log('Previous animation ended, finalizing page positions...');
                
                // Clean up animation classes and clear transform
                currentLeftPage.classList.remove('page-flip', 'reverse', 'flipping');
                currentLeftPage.style.transform = '';
                
                // Get the current right page BEFORE we change classes
                const currentRightPage = container.querySelector('.current-right');
                console.log('Current right page before changes:', currentRightPage);
                
                // Set up the new left page (prevLeftPageElement)
                prevLeftPageElement.classList.remove('right-page');
                prevLeftPageElement.classList.add('current-left');
                
                // CRITICAL: Disable only positioning transitions to prevent scattering, keep smooth transforms
                prevLeftPageElement.style.setProperty('transition', 'transform 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94), opacity 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94)', 'important');
                
                prevLeftPageElement.style.display = 'block';
                prevLeftPageElement.style.visibility = 'visible';
                prevLeftPageElement.style.opacity = '1';
                prevLeftPageElement.style.zIndex = '5';
                prevLeftPageElement.style.pointerEvents = 'auto';
                prevLeftPageElement.style.backfaceVisibility = 'visible';
                prevLeftPageElement.style.webkitBackfaceVisibility = 'visible';
                prevLeftPageElement.style.transform = '';
                // Force explicit positioning for left page with !important
                prevLeftPageElement.style.setProperty('position', 'absolute', 'important');
                prevLeftPageElement.style.setProperty('left', '0%', 'important');
                prevLeftPageElement.style.setProperty('right', 'auto', 'important');
                prevLeftPageElement.style.setProperty('width', '50%', 'important');
                prevLeftPageElement.style.setProperty('top', '1%', 'important');
                prevLeftPageElement.style.setProperty('height', '98%', 'important');
                prevLeftPageElement.style.setProperty('transform', 'rotateY(-1deg) translateZ(0px) translateX(-0.5px)', 'important');
                prevLeftPageElement.style.setProperty('transform-origin', 'right center', 'important');
                
                // Ensure the left page image is properly loaded and visible
                const prevLeftPageImg = prevLeftPageElement.querySelector('img');
                if (prevLeftPageImg) {
                    prevLeftPageImg.style.display = 'block';
                    prevLeftPageImg.style.visibility = 'visible';
                    prevLeftPageImg.style.opacity = '1';
                    prevLeftPageImg.style.backfaceVisibility = 'visible';
                    prevLeftPageImg.style.webkitBackfaceVisibility = 'visible';
                    
                    // Force image reload if it's not loaded
                    if (!prevLeftPageImg.complete || prevLeftPageImg.naturalWidth === 0) {
                        console.log('Previous left page image not loaded, forcing reload...');
                        const originalSrc = prevLeftPageImg.src;
                        prevLeftPageImg.src = '';
                        prevLeftPageImg.src = originalSrc;
                    }
                }
                
                console.log('Set up new left page:', prevLeftPageNumber);
                
                // Set up the new right page (prevRightPageElement)
                prevRightPageElement.classList.remove('right-page');
                prevRightPageElement.classList.add('current-right');
                
                // CRITICAL: Disable only positioning transitions to prevent scattering, keep smooth transforms
                prevRightPageElement.style.setProperty('transition', 'transform 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94), opacity 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94)', 'important');
                
                prevRightPageElement.style.display = 'block';
                prevRightPageElement.style.visibility = 'visible';
                prevRightPageElement.style.opacity = '1';
                prevRightPageElement.style.zIndex = '5';
                prevRightPageElement.style.pointerEvents = 'auto';
                prevRightPageElement.style.backfaceVisibility = 'visible';
                prevRightPageElement.style.webkitBackfaceVisibility = 'visible';
                prevRightPageElement.style.transform = '';
                // Force explicit positioning for right page - attach to spine
                prevRightPageElement.style.setProperty('position', 'absolute', 'important');
                prevRightPageElement.style.setProperty('left', 'auto', 'important');
                prevRightPageElement.style.setProperty('right', '0%', 'important');
                prevRightPageElement.style.setProperty('width', '50%', 'important');
                prevRightPageElement.style.setProperty('top', '1%', 'important');
                prevRightPageElement.style.setProperty('height', '98%', 'important');
                prevRightPageElement.style.setProperty('margin-left', '0', 'important');
                prevRightPageElement.style.setProperty('margin-right', '0', 'important');
                prevRightPageElement.style.setProperty('padding-left', '0', 'important');
                prevRightPageElement.style.setProperty('padding-right', '0', 'important');
                prevRightPageElement.style.setProperty('box-sizing', 'border-box', 'important');
                prevRightPageElement.style.setProperty('transform', 'rotateY(1deg) translateZ(0px) translateX(0.5px)', 'important');
                prevRightPageElement.style.setProperty('transform-origin', 'left center', 'important');
                
                // CRITICAL: Additional enforcement to ensure right page stays on right side
                prevRightPageElement.style.setProperty('float', 'none', 'important');
                prevRightPageElement.style.setProperty('clear', 'none', 'important');
                prevRightPageElement.style.setProperty('display', 'block', 'important');
                prevRightPageElement.style.setProperty('position', 'absolute', 'important');
                
                // CRITICAL: Enforce right page visibility immediately
                if (typeof window.enforceRightPageVisibility === 'function') {
                    window.enforceRightPageVisibility(container.id);
                }
                
                // Ensure the right page image is properly loaded and visible
                const prevRightPageImg = prevRightPageElement.querySelector('img');
                if (prevRightPageImg) {
                    prevRightPageImg.style.display = 'block';
                    prevRightPageImg.style.visibility = 'visible';
                    prevRightPageImg.style.opacity = '1';
                    prevRightPageImg.style.backfaceVisibility = 'visible';
                    prevRightPageImg.style.webkitBackfaceVisibility = 'visible';
                    
                    // Force image reload if it's not loaded
                    if (!prevRightPageImg.complete || prevRightPageImg.naturalWidth === 0) {
                        console.log('Previous right page image not loaded, forcing reload...');
                        const originalSrc = prevRightPageImg.src;
                        prevRightPageImg.src = '';
                        prevRightPageImg.src = originalSrc;
                    }
                }
                
                console.log('Set up new right page:', prevRightPageNumber);
                
                // CRITICAL: Smooth positioning transition that feels natural
                setTimeout(() => {
                    // Add smooth positioning transition with realistic bounce effect
                    prevLeftPageElement.style.setProperty('transition', 'left 0.6s cubic-bezier(0.34, 1.56, 0.64, 1), right 0.6s cubic-bezier(0.34, 1.56, 0.64, 1), transform 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94), opacity 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94)', 'important');
                    prevRightPageElement.style.setProperty('transition', 'left 0.6s cubic-bezier(0.34, 1.56, 0.64, 1), right 0.6s cubic-bezier(0.34, 1.56, 0.64, 1), transform 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94), opacity 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94)', 'important');
                    
                    // Re-enforce left page positioning with smooth transition (settles first)
                    prevLeftPageElement.style.setProperty('left', '0%', 'important');
                    prevLeftPageElement.style.setProperty('right', 'auto', 'important');
                    prevLeftPageElement.style.setProperty('transform', 'rotateY(-1deg) translateZ(0px) translateX(-0.5px)', 'important');
                    
                    // Stagger the right page positioning for more natural feel
                    setTimeout(() => {
                        // Re-enforce right page positioning with smooth transition (settles second)
                        prevRightPageElement.style.setProperty('left', 'auto', 'important');
                        prevRightPageElement.style.setProperty('right', '0%', 'important');
                        prevRightPageElement.style.setProperty('transform', 'rotateY(1deg) translateZ(0px) translateX(0.5px)', 'important');
                        prevRightPageElement.style.setProperty('position', 'absolute', 'important');
                        prevRightPageElement.style.setProperty('width', '50%', 'important');
                        prevRightPageElement.style.setProperty('float', 'none', 'important');
                        prevRightPageElement.style.setProperty('clear', 'none', 'important');
                    }, 50); // Small delay for natural stagger effect
                    
                    console.log('Applied smooth positioning transition');
                    
                    // After positioning transition completes, restore full transitions
                    setTimeout(() => {
                        prevLeftPageElement.style.setProperty('transition', 'all 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94)', 'important');
                        prevRightPageElement.style.setProperty('transition', 'all 1.4s cubic-bezier(0.25, 0.46, 0.45, 0.94)', 'important');
                        console.log('Restored full CSS transitions for future interactions');
                    }, 700); // Wait for positioning transition to complete (0.6s + buffer)
                }, 100);
                
                // Ensure all pages have proper pointer events after animation
                setTimeout(() => {
                    const allPages = container.querySelectorAll('.dearflip-page');
                    allPages.forEach(page => {
                        if (page.classList.contains('current-left') || page.classList.contains('current-right')) {
                            page.style.pointerEvents = 'auto';
                        }
                    });
                    console.log('Ensured all current pages have pointer events enabled (previous)');
                }, 100);
                
                // Hide the old current left page (it was flipping) using deferred updates
                currentLeftPage.classList.remove('current-left', 'page-flip', 'reverse', 'flipping');
                currentLeftPage.classList.add('right-page');
                this.deferVisibilityUpdate(currentLeftPage, 'hide');
                currentLeftPage.style.zIndex = '1';
                currentLeftPage.style.transform = '';
                
                // Remove the backface content that was added during the flip
                const backfaceContent = currentLeftPage.querySelector('[style*="rotateY(180deg)"]');
                if (backfaceContent) {
                    backfaceContent.remove();
                }
                
                console.log('Hidden old current left page and cleaned up backface');
                
                // Hide the old current right page using deferred updates
                if (currentRightPage && currentRightPage !== currentLeftPage) {
                    currentRightPage.classList.remove('current-right');
                    currentRightPage.classList.add('right-page');
                    this.deferVisibilityUpdate(currentRightPage, 'hide');
                    currentRightPage.style.zIndex = '1';
                    console.log('Hidden old current right page');
                }
                
                instance.currentPage -= 2; // Decrement by 2 since we show 2 pages at a time
                
                // Update button states after page flip
                this.updateButtonStates(container);
                
                // Trigger page info update if callback exists
                if (typeof window.updatePageInfo === 'function') {
                    setTimeout(() => window.updatePageInfo(), 100);
                }
                
                console.log(`Previous page flip complete. Now showing pages ${instance.currentPage - 1} and ${instance.currentPage}`);
                
                // Reset flipping flag and timing
                instance.currentPage = prevLeftPageNumber + 1;
                instance.isFlipping = false;
                instance.flipStartTime = null;
                instance.flipController = null;
                
                // Step 5: Silently upgrade visible pages to high quality after flip completes
                // Run in background without blocking or triggering events
                setTimeout(() => {
                    this.upgradeVisiblePagesQuality(container, [prevLeftPageNumber, prevRightPageNumber]);
                }, 100);
                
                // Trigger predictive preloading for next pages
                this.predictivePreload(container.id);
                
                // Remove the event listener to prevent memory leaks
                currentLeftPage.removeEventListener('animationend', handlePreviousAnimationEnd);
                currentLeftPage.removeEventListener('webkitAnimationEnd', handlePreviousAnimationEnd);
            };
            
            // Add event listeners for animation end
            currentLeftPage.addEventListener('animationend', handlePreviousAnimationEnd);
            currentLeftPage.addEventListener('webkitAnimationEnd', handlePreviousAnimationEnd);
            
            // Enhanced fallback timeout with multiple checks
            setTimeout(() => {
                if (!previousAnimationCompleted) {
                    console.log('Fallback timeout reached for previous page, ensuring animation completion...');
                    if (currentLeftPage.classList.contains('page-flip')) {
                        handlePreviousAnimationEnd();
                    }
                } else {
                    console.log('Previous animation already completed, skipping fallback timeout');
                }
            }, 1500); // Slightly longer than animation duration
            
            // Additional safety check - force reset if stuck
            setTimeout(() => {
                if (instance.isFlipping) {
                    console.warn('CRITICAL: Previous flip appears stuck, forcing reset...');
                    instance.isFlipping = false;
                    // Clean up any stuck animation classes
                    currentLeftPage.classList.remove('page-flip', 'reverse', 'flipping');
                    currentLeftPage.style.transform = '';
                }
            }, 3000); // Emergency reset after 3 seconds
        } else {
            console.error('Cannot flip to previous page - missing elements:');
            console.error('currentLeftPage:', currentLeftPage);
            console.error('prevPageElement:', prevPageElement);
        }
    }

    /**
     * Go to previous page
     */
    async previousPage(containerId = null) {
        console.log('previousPage called with containerId:', containerId);
        const container = containerId ? document.getElementById(containerId) : this.getContainerFromEvent();
        console.log('Container found:', container);
        console.log('Container ID:', container?.id);
        if (!container) {
            console.error('Container not found for previousPage');
            return;
        }
        
        const instance = this.instances.get(container.id);
        if (instance) {
            console.log(`Previous page - Current: ${instance.currentPage}, Total: ${instance.totalPages}`);
            
            // CRITICAL: Check if already flipping to prevent race conditions
            if (instance.isFlipping) {
                console.log('Previous page ignored - already flipping');
                return;
            }
            
            // CRITICAL: Check if we can go to previous page
            if (instance.currentPage <= 1) {
                console.log('Already at first page, cannot go previous');
                return;
            }
        }
        
        console.log('Calling flipToPreviousPage...');
        await this.flipToPreviousPage(container);
    }

    /**
     * Go to next page
     */
    async nextPage(containerId = null) {
        console.log('nextPage called with containerId:', containerId);
        const container = containerId ? document.getElementById(containerId) : this.getContainerFromEvent();
        console.log('Container found:', container);
        console.log('Container ID:', container?.id);
        if (!container) {
            console.error('Container not found for nextPage');
            return;
        }
        
        const instance = this.instances.get(container.id);
        if (instance) {
            console.log(`Next page - Current: ${instance.currentPage}, Total: ${instance.totalPages}`);
            
            // CRITICAL: Check if already flipping to prevent race conditions
            if (instance.isFlipping) {
                console.log('Next page ignored - already flipping');
                return;
            }
            
            // CRITICAL: Check if we can go to next page
            if (instance.currentPage >= instance.totalPages) {
                console.log('Already at last page, cannot go next');
                return;
            }
        }
        
        console.log('Calling flipToNextPage...');
        await this.flipToNextPage(container);
    }

    /**
     * Get container from event context
     */
    getContainerFromEvent() {
        // This is a simplified approach - in a real implementation,
        // you'd pass the container ID or use a more sophisticated method
        const containers = document.querySelectorAll('.dearflip-container');
        return containers[0]; // For demo purposes, use the first container
    }

    /**
     * EMERGENCY FIX: Force check and fix any page rendering issues
     * This function can be called manually if pages don't render
     */
    forceFixPageRendering(containerId = null) {
        console.log('EMERGENCY: Force fixing page rendering...');
        const container = containerId ? document.getElementById(containerId) : this.getContainerFromEvent();
        if (!container) {
            console.error('No container found for emergency fix');
            return;
        }

        const currentLeftPage = container.querySelector('.current-left');
        const currentRightPage = container.querySelector('.current-right');

        if (currentLeftPage) {
            console.log('Fixing left page...');
            currentLeftPage.style.display = 'block';
            currentLeftPage.style.visibility = 'visible';
            currentLeftPage.style.opacity = '1';
            currentLeftPage.style.zIndex = '5';
            
            const leftImg = currentLeftPage.querySelector('img');
            if (leftImg) {
                leftImg.style.display = 'block';
                leftImg.style.visibility = 'visible';
                leftImg.style.opacity = '1';
                if (!leftImg.complete || leftImg.naturalWidth === 0) {
                    leftImg.style.background = 'white';
                }
            }
        }

        if (currentRightPage) {
            console.log('Fixing right page...');
            currentRightPage.style.display = 'block';
            currentRightPage.style.visibility = 'visible';
            currentRightPage.style.opacity = '1';
            currentRightPage.style.zIndex = '5';
            
            const rightImg = currentRightPage.querySelector('img');
            if (rightImg) {
                rightImg.style.display = 'block';
                rightImg.style.visibility = 'visible';
                rightImg.style.opacity = '1';
                if (!rightImg.complete || rightImg.naturalWidth === 0) {
                    rightImg.style.background = 'white';
                }
            }
        }

        console.log('Emergency fix completed');
    }

    /**
     * Update page information display
     */
    updatePageInfo(container, currentPage, totalPages) {
        // This method can be used to update any page information display
        // Currently not displaying page info, but keeping for future use
        console.log(`Page info: ${currentPage} of ${totalPages}`);
    }

    /**
     * Update navigation button states based on current page
     */
    updateButtonStates(container) {
        const instance = this.instances.get(container.id);
        if (!instance) {
            console.error('No instance found for button state update');
            return;
        }

        // Look for navigation buttons in the document (they're positioned outside the container)
        const prevBtn = document.querySelector('#prev-btn');
        const nextBtn = document.querySelector('#next-btn');

        if (!prevBtn || !nextBtn) {
            console.error('Navigation buttons not found');
            return;
        }

        console.log(`Updating button states - Current page: ${instance.currentPage}, Total pages: ${instance.totalPages}`);

        // Disable previous button if we're at the first page
        if (instance.totalPages === 1) {
            // Single page document - always disable previous button
            prevBtn.disabled = true;
            prevBtn.style.opacity = '0.3';
            prevBtn.style.cursor = 'not-allowed';
            console.log('Previous button disabled - single page document');
        } else if (instance.currentPage <= 2) {
            // Multi-page document - disable if showing pages 1-2
            prevBtn.disabled = true;
            prevBtn.style.opacity = '0.3';
            prevBtn.style.cursor = 'not-allowed';
            console.log('Previous button disabled - at first page');
        } else {
            prevBtn.disabled = false;
            prevBtn.style.opacity = '1';
            prevBtn.style.cursor = 'pointer';
            console.log('Previous button enabled');
        }

        // Disable next button if we're at the last page
        if (instance.totalPages === 1) {
            // Single page document - always disable next button
            nextBtn.disabled = true;
            nextBtn.style.opacity = '0.3';
            nextBtn.style.cursor = 'not-allowed';
            console.log('Next button disabled - single page document');
        } else if (instance.currentPage >= instance.totalPages) {
            // Multi-page document - disable if at last page
            nextBtn.disabled = true;
            nextBtn.style.opacity = '0.3';
            nextBtn.style.cursor = 'not-allowed';
            console.log('Next button disabled - at last page');
        } else {
            nextBtn.disabled = false;
            nextBtn.style.opacity = '1';
            nextBtn.style.cursor = 'pointer';
            console.log('Next button enabled');
        }
    }


    /**
     * Get instance by container ID
     */
    static getInstance(containerId) {
        if (!window.DearFlipEngineInstance) {
            window.DearFlipEngineInstance = new DearFlipEngine();
        }
        return window.DearFlipEngineInstance.instances.get(containerId);
    }

    /**
     * Upgrade visible pages from low quality to high quality (silently, no loader)
     * Step 5: Silent quality upgrade method
     */
    async upgradeVisiblePagesQuality(container, pageNumbers) {
        const instance = this.instances.get(container.id);
        if (!instance) return;

        // Upgrade silently in background without any loader
        try {
            const upgradePromises = pageNumbers.map(async (pageNumber) => {
                try {
                    const pageElement = container.querySelector(`[data-page-number="${pageNumber}"]`);
                    if (!pageElement) {
                        console.log(`Page ${pageNumber} element not found, skipping upgrade`);
                        return;
                    }

                    const img = pageElement.querySelector('img');
                    if (!img) {
                        console.log(`Page ${pageNumber} image not found, skipping upgrade`);
                        return;
                    }

                    // Check if page is already high quality
                    const pageData = instance.allPages[pageNumber - 1];
                    if (pageData && pageData.quality === 'high') {
                        return; // Already high quality
                    }

                    // Upgrade to high quality silently
                    const highQualityData = await this.pdfRenderer.upgradePageQuality(pageNumber);
                    
                    if (highQualityData) {
                        // Update the image source with high quality version
                        // Use a new Image object to prevent triggering events on the existing img
                        const newImg = new Image();
                        newImg.onload = () => {
                            // Only update if the page element still exists and hasn't been removed
                            const currentPageElement = container.querySelector(`[data-page-number="${pageNumber}"]`);
                            if (currentPageElement && currentPageElement.contains(img)) {
                                img.src = highQualityData.dataURL;
                                
                                // Update instance data
                                instance.allPages[pageNumber - 1] = highQualityData;
                                
                                // Smooth fade-in transition for high quality
                                img.style.transition = 'opacity 0.2s ease-in';
                                img.style.opacity = '0.95';
                                setTimeout(() => {
                                    img.style.opacity = '1';
                                }, 50);
                                
                                console.log(`Page ${pageNumber} silently upgraded to high quality`);
                            }
                        };
                        newImg.src = highQualityData.dataURL;
                    }
                } catch (error) {
                    console.error(`Error upgrading page ${pageNumber}:`, error);
                    // Fail silently - don't interrupt user experience
                }
            });

            await Promise.all(upgradePromises);
        } catch (error) {
            console.error('Error during quality upgrade:', error);
            // Fail silently - don't interrupt user experience
        }
    }
}

// Global instance
console.log('Creating global DearFlipEngine instance...');
window.DearFlipEngine = new DearFlipEngine();
console.log('Global DearFlipEngine instance created:', window.DearFlipEngine);

// Test method to verify the global instance is working
window.DearFlipEngine.test = function() {
    console.log('DearFlipEngine global instance is working!');
    return true;
};

// Test the instance immediately
window.DearFlipEngine.test();

// EMERGENCY FIX: Global function to force fix page rendering
window.forceFixPageRendering = function(containerId = null) {
    if (window.DearFlipEngine) {
        window.DearFlipEngine.forceFixPageRendering(containerId);
    } else {
        console.error('DearFlipEngine not available');
    }
};

// Auto-fix every 2 seconds as a safety net
setInterval(() => {
    const containers = document.querySelectorAll('.dearflip-container');
    containers.forEach(container => {
        const currentRightPage = container.querySelector('.current-right');
        if (currentRightPage) {
            const rightImg = currentRightPage.querySelector('img');
            
            // Check for right page visibility issues
            const isRightPageVisible = currentRightPage.style.display !== 'none' && 
                                     currentRightPage.style.visibility !== 'hidden' && 
                                     currentRightPage.style.opacity !== '0';
            
            const isRightImgVisible = rightImg && 
                                    rightImg.style.display !== 'none' && 
                                    rightImg.style.visibility !== 'hidden' && 
                                    rightImg.style.opacity !== '0';
            
            const isRightImgLoaded = rightImg && rightImg.complete && rightImg.naturalWidth > 0;
            
            if (!isRightPageVisible || !isRightImgVisible || !isRightImgLoaded) {
                console.log('AUTO-FIX: Detected right page visibility issue, applying fix...');
                console.log('- Right page visible:', isRightPageVisible);
                console.log('- Right img visible:', isRightImgVisible);
                console.log('- Right img loaded:', isRightImgLoaded);
                
                // Use the specific right page enforcement function
                if (typeof window.enforceRightPageVisibility === 'function') {
                    window.enforceRightPageVisibility(container.id);
                } else {
                    window.forceFixPageRendering(container.id);
                }
            }
        }
    });
}, 2000);

// EMERGENCY: Manual page reload function
window.reloadCurrentPages = function(containerId = null) {
    console.log('EMERGENCY: Reloading current pages...');
    const container = containerId ? document.getElementById(containerId) : document.querySelector('.dearflip-container');
    if (!container) {
        console.error('No container found for page reload');
        return;
    }

    const instance = window.DearFlipEngine.instances.get(container.id);
    if (!instance) {
        console.error('No instance found for page reload');
        return;
    }

    const currentLeftPage = container.querySelector('.current-left');
    const currentRightPage = container.querySelector('.current-right');

    if (currentLeftPage) {
        const leftImg = currentLeftPage.querySelector('img');
        if (leftImg) {
            const originalSrc = leftImg.src;
            leftImg.src = '';
            leftImg.src = originalSrc + '?reload=' + Date.now();
            console.log('Reloaded left page image');
        }
    }

    if (currentRightPage) {
        const rightImg = currentRightPage.querySelector('img');
        if (rightImg) {
            const originalSrc = rightImg.src;
            rightImg.src = '';
            rightImg.src = originalSrc + '?reload=' + Date.now();
            console.log('Reloaded right page image');
        }
    }

    console.log('Emergency page reload completed');
};

// EMERGENCY: Force flip retry function
window.forceFlipRetry = function(containerId = null) {
    console.log('EMERGENCY: Force flip retry...');
    const container = containerId ? document.getElementById(containerId) : document.querySelector('.dearflip-container');
    if (!container) {
        console.error('No container found for flip retry');
        return;
    }

    const instance = window.DearFlipEngine.instances.get(container.id);
    if (instance) {
        instance.isFlipping = false;
        console.log('Reset flipping flag, you can now try flipping again');
    }
};

// CRITICAL: Right page visibility enforcement
window.enforceRightPageVisibility = function(containerId = null) {
    console.log('CRITICAL: Enforcing right page visibility...');
    const container = containerId ? document.getElementById(containerId) : document.querySelector('.dearflip-container');
    if (!container) {
        console.error('No container found for right page enforcement');
        return;
    }

    const currentRightPage = container.querySelector('.current-right');
    if (!currentRightPage) {
        console.error('No current right page found');
        return;
    }

    const rightImg = currentRightPage.querySelector('img');
    if (!rightImg) {
        console.error('No right page image found');
        return;
    }

    // FORCE right page visibility
    currentRightPage.style.display = 'block';
    currentRightPage.style.visibility = 'visible';
    currentRightPage.style.opacity = '1';
    currentRightPage.style.zIndex = '5';
    currentRightPage.style.pointerEvents = 'auto';
    currentRightPage.style.position = 'relative';
    currentRightPage.style.transform = 'none';
    currentRightPage.style.backfaceVisibility = 'visible';
    currentRightPage.style.webkitBackfaceVisibility = 'visible';

    // FORCE right image visibility
    rightImg.style.display = 'block';
    rightImg.style.visibility = 'visible';
    rightImg.style.opacity = '1';
    rightImg.style.width = '100%';
    rightImg.style.height = '100%';
    rightImg.style.objectFit = 'contain';
    rightImg.style.position = 'relative';
    rightImg.style.zIndex = '1';

    // If image is not loaded, force reload
    if (!rightImg.complete || rightImg.naturalWidth === 0) {
        console.log('Right page image not loaded, forcing reload...');
        const originalSrc = rightImg.src;
        rightImg.src = '';
        rightImg.src = originalSrc + '?force=' + Date.now();
        
        rightImg.onload = () => {
            console.log('Right page image reloaded successfully');
        };
        
        rightImg.onerror = () => {
            console.error('Right page image failed to reload');
            // Set white background as fallback
            rightImg.style.background = 'white';
            rightImg.style.minHeight = '400px';
            rightImg.style.minWidth = '300px';
        };
    }

    console.log('Right page visibility enforced');
};

// Quick fix for scattered pages after reverse flip
window.fixScatteredPages = function(containerId) {
    const container = document.getElementById(containerId);
    if (!container) return;
    
    const leftPage = container.querySelector('.current-left');
    const rightPage = container.querySelector('.current-right');
    
    if (leftPage) {
        leftPage.style.setProperty('left', '0%', 'important');
        leftPage.style.setProperty('right', 'auto', 'important');
        leftPage.style.setProperty('transform', 'rotateY(-1deg) translateZ(0px) translateX(-0.5px)', 'important');
    }
    
    if (rightPage) {
        rightPage.style.setProperty('left', 'auto', 'important');
        rightPage.style.setProperty('right', '0%', 'important');
        rightPage.style.setProperty('transform', 'rotateY(1deg) translateZ(0px) translateX(0.5px)', 'important');
        rightPage.style.setProperty('position', 'absolute', 'important');
        rightPage.style.setProperty('width', '50%', 'important');
        rightPage.style.setProperty('float', 'none', 'important');
        rightPage.style.setProperty('clear', 'none', 'important');
        rightPage.style.setProperty('display', 'block', 'important');
    }
    
    console.log('Fixed scattered pages');
};

// Specific function to force right page to right side
window.fixRightPagePosition = function(containerId) {
    const container = document.getElementById(containerId);
    if (!container) return;
    
    const rightPage = container.querySelector('.current-right');
    if (!rightPage) {
        console.error('No right page found');
        return;
    }
    
    // Force right page to right side of spine
    rightPage.style.setProperty('left', 'auto', 'important');
    rightPage.style.setProperty('right', '0%', 'important');
    rightPage.style.setProperty('position', 'absolute', 'important');
    rightPage.style.setProperty('width', '50%', 'important');
    rightPage.style.setProperty('top', '1%', 'important');
    rightPage.style.setProperty('height', '98%', 'important');
    rightPage.style.setProperty('transform', 'rotateY(1deg) translateZ(0px) translateX(0.5px)', 'important');
    rightPage.style.setProperty('transform-origin', 'left center', 'important');
    rightPage.style.setProperty('float', 'none', 'important');
    rightPage.style.setProperty('clear', 'none', 'important');
    rightPage.style.setProperty('display', 'block', 'important');
    rightPage.style.setProperty('margin-left', '0', 'important');
    rightPage.style.setProperty('margin-right', '0', 'important');
    rightPage.style.setProperty('padding-left', '0', 'important');
    rightPage.style.setProperty('padding-right', '0', 'important');
    rightPage.style.setProperty('box-sizing', 'border-box', 'important');
    
    console.log('Forced right page to right side of spine');
};

// Emergency function to fix stuck flips
window.fixStuckFlip = function(containerId) {
    const container = document.getElementById(containerId);
    if (!container) {
        console.error(`Container ${containerId} not found`);
        return false;
    }
    
    const instance = window.DearFlipEngine.instances.get(container.id);
    if (!instance) {
        console.error('No instance found');
        return false;
    }
    
    // Force reset flipping state
    instance.isFlipping = false;
    
    // Clean up any stuck animation classes
    const allPages = container.querySelectorAll('.dearflip-page');
    allPages.forEach(page => {
        page.classList.remove('page-flip', 'reverse', 'flipping');
        page.style.transform = '';
    });
    
    // Ensure current pages are visible
    const currentLeftPage = container.querySelector('.current-left');
    const currentRightPage = container.querySelector('.current-right');
    
    if (currentLeftPage) {
        currentLeftPage.style.display = 'block';
        currentLeftPage.style.visibility = 'visible';
        currentLeftPage.style.opacity = '1';
    }
    
    if (currentRightPage) {
        currentRightPage.style.display = 'block';
        currentRightPage.style.visibility = 'visible';
        currentRightPage.style.opacity = '1';
    }
    
    console.log('Fixed stuck flip - you can now flip again');
    return true;
};

// Cache management utilities
window.getCacheStats = function() {
    if (window.DearFlipEngine && window.DearFlipEngine.pdfRenderer) {
        return window.DearFlipEngine.pdfRenderer.getCacheStats();
    }
    return null;
};

window.clearCache = function() {
    if (window.DearFlipEngine && window.DearFlipEngine.pdfRenderer) {
        window.DearFlipEngine.pdfRenderer.clearCache();
        console.log('Cache cleared manually');
    }
};

window.saveCache = async function() {
    if (window.DearFlipEngine && window.DearFlipEngine.pdfRenderer) {
        await window.DearFlipEngine.pdfRenderer.saveCachedPages();
        console.log('Cache saved manually');
    }
};

// Auto-detect and fix stuck flips every 5 seconds
setInterval(() => {
    const containers = document.querySelectorAll('.dearflip-container');
    containers.forEach(container => {
        const instance = window.DearFlipEngine.instances.get(container.id);
        if (instance && instance.isFlipping) {
            // Check if flip has been stuck for more than 5 seconds
            const now = Date.now();
            if (!instance.flipStartTime) {
                instance.flipStartTime = now;
            } else if (now - instance.flipStartTime > 5000) {
                console.warn('Auto-detected stuck flip, fixing...');
                window.fixStuckFlip(container.id);
                instance.flipStartTime = null;
            }
        } else if (instance) {
            instance.flipStartTime = null;
        }
    });
}, 5000);

// Save cache when page is unloaded
window.addEventListener('beforeunload', () => {
    if (window.DearFlipEngine && window.DearFlipEngine.pdfRenderer) {
        window.DearFlipEngine.pdfRenderer.saveCachedPages();
    }
});

// Save cache periodically (every 30 seconds)
setInterval(() => {
    if (window.DearFlipEngine && window.DearFlipEngine.pdfRenderer) {
        window.DearFlipEngine.pdfRenderer.saveCachedPages();
    }
}, 30000);

// Enhanced text sharpness utilities
window.setTextSharpnessMode = function(mode) {
    if (window.DearFlipEngine) {
        window.DearFlipEngine.setTextSharpnessMode(mode);
    }
};

window.setTextLayerEnabled = function(enabled) {
    if (window.DearFlipEngine) {
        window.DearFlipEngine.setTextLayerEnabled(enabled);
    }
};

window.handleZoomEvent = function(zoomLevel) {
    if (window.DearFlipEngine) {
        window.DearFlipEngine.handleZoomEvent(zoomLevel);
    }
};

// Performance monitoring and debug utilities
window.enablePerformanceMonitoring = function() {
    console.log('Enabling performance monitoring...');
    
    // Enable GPU compositing hints on body
    document.body.style.setProperty('transform', 'translateZ(0)');
    document.body.style.setProperty('will-change', 'transform');
    
    // Monitor frame rate
    let frameCount = 0;
    let lastTime = performance.now();
    
    const monitorFrameRate = () => {
        frameCount++;
        const currentTime = performance.now();
        if (currentTime - lastTime >= 1000) {
            const fps = Math.round((frameCount * 1000) / (currentTime - lastTime));
            console.log(`FPS: ${fps}`);
            frameCount = 0;
            lastTime = currentTime;
        }
        requestAnimationFrame(monitorFrameRate);
    };
    
    requestAnimationFrame(monitorFrameRate);
    console.log('Performance monitoring enabled. Check console for FPS.');
};

// Debug function to check GPU layer status
window.checkGPULayers = function() {
    const containers = document.querySelectorAll('.dearflip-container');
    containers.forEach((container, index) => {
        const pages = container.querySelectorAll('.dearflip-page');
        console.log(`Container ${index}:`);
        pages.forEach((page, pageIndex) => {
            const computedStyle = window.getComputedStyle(page);
            console.log(`  Page ${pageIndex}:`, {
                willChange: computedStyle.willChange,
                transform: computedStyle.transform,
                backfaceVisibility: computedStyle.backfaceVisibility,
                translate: computedStyle.translate
            });
        });
    });
};

// Auto-enable performance monitoring in development
if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
    console.log('Development mode detected. Use enablePerformanceMonitoring() to monitor performance.');
}

