document.addEventListener('DOMContentLoaded', function() {
    const searchForm = document.querySelector('.top-search-form');
    const searchInput = searchForm.querySelector('input[name="q"]');

    document.addEventListener('keydown', function(event) {
        if (event.key === 'Enter') {
            event.preventDefault();
            return false;
        }
    });

    const searchResults = document.createElement('div');
    searchResults.id = 'searchResults';
    searchResults.className = 'search-results';
    searchForm.appendChild(searchResults);
    
    let searchTimeout;

    if (searchInput) {
        searchInput.addEventListener('input', function(e) {
            const query = e.target.value.trim();
            
            // Clear previous timeout
            clearTimeout(searchTimeout);
            
            // Hide results if query is empty
            if (!query) {
                searchResults.style.display = 'none';
                return;
            }

            // Set a timeout to prevent too many requests
            searchTimeout = setTimeout(() => {
                fetch(`/search?query=${encodeURIComponent(query)}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.error) {
                            searchResults.innerHTML = `<div class="search-error">${data.error}</div>`;
                            searchResults.style.display = 'block';
                            return;
                        }

                        if (data.length > 0) {
                            searchResults.innerHTML = data.map(item => {
                                if (item.type === 'news') {
                                    return `
                                        <div class="search-result-item news-result">
                                            <a href="${item.url}" class="d-flex align-items-center">
                                                <div class="search-result-content">
                                                    <h6 class="search-result-title">${item.title}</h6>
                                                    <p class="search-result-description">${item.description}</p>
                                                    ${item.date ? `<small class="search-result-date">${item.date}</small>` : ''}
                                                    <div class="search-result-matches">
                                                        <span class="match-type news">News match</span>
                                                    </div>
                                                </div>
                                            </a>
                                        </div>
                                    `;
                                } else if (item.type === 'page') {
                                    return `
                                        <div class="search-result-item page-result">
                                            <a href="${item.url}" class="d-flex align-items-center">
                                                <div class="search-result-content">
                                                    <h6 class="search-result-title">${item.title}</h6>
                                                    <p class="search-result-description">${item.description}</p>
                                                    ${item.matches && item.matches.length > 0 ? 
                                                        `<div class="search-result-matches">
                                                            ${item.matches.map(match => `
                                                                <span class="match-type ${match.type}">
                                                                    ${match.type === 'title' ? 'Title' : 
                                                                      match.type === 'heading' ? 'Heading' : 
                                                                      'Content'} match
                                                                </span>
                                                            `).join('')}
                                                        </div>` : ''}
                                                </div>
                                            </a>
                                        </div>
                                    `;
                                }
                            }).join('');
                            searchResults.style.display = 'block';
                        } else {
                            searchResults.innerHTML = '<div class="search-no-results">No results found</div>';
                            searchResults.style.display = 'block';
                        }
                    })
                    .catch(error => {
                        console.error('Search error:', error);
                        searchResults.innerHTML = '<div class="search-error">An error occurred while searching</div>';
                        searchResults.style.display = 'block';
                    });
            }, 300); // 300ms delay
        });

        // Handle form submission
        searchForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const query = searchInput.value.trim();
            if (query) {
                window.location.href = `/search?query=${encodeURIComponent(query)}`;
            }
        });

        // Hide results when clicking outside
        document.addEventListener('click', function(e) {
            if (!searchForm.contains(e.target)) {
                searchResults.style.display = 'none';
            }
        });
    }


    document.addEventListener('keydown', function(event) {
        if (event.key === 'Enter') {
            event.preventDefault();
            return false;
        }
    });
}); 