@extends('layouts.app')

@section('content')
<div class="container mt-4">
    <div class="row">
        <div class="col-12">
            <h2>Spam Monitoring Dashboard</h2>
            <p class="text-muted">Monitor and manage spam attempts on your forms</p>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <h5 class="card-title">Total Attempts</h5>
                    <h3>{{ $stats['total_attempts'] ?? 0 }}</h3>
                    <small>Last {{ $days }} days</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <h5 class="card-title">Blocked Attempts</h5>
                    <h3>{{ $stats['blocked_attempts'] ?? 0 }}</h3>
                    <small>Successfully blocked</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-info text-white">
                <div class="card-body">
                    <h5 class="card-title">Contact Form</h5>
                    <h3>{{ $stats['by_form']['contact'] ?? 0 }}</h3>
                    <small>Spam attempts</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-warning text-white">
                <div class="card-body">
                    <h5 class="card-title">Career Form</h5>
                    <h3>{{ $stats['by_form']['career'] ?? 0 }}</h3>
                    <small>Spam attempts</small>
                </div>
            </div>
        </div>
    </div>

    <!-- Spam Type Breakdown -->
    <div class="row mb-4">
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5>Spam Types (Last {{ $days }} days)</h5>
                </div>
                <div class="card-body">
                    @if(isset($stats['by_type']) && count($stats['by_type']) > 0)
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Type</th>
                                        <th>Count</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($stats['by_type'] as $type => $count)
                                    <tr>
                                        <td>
                                            @switch($type)
                                                @case('honeypot')
                                                    <span class="badge bg-danger">Honeypot</span>
                                                    @break
                                                @case('timing')
                                                    <span class="badge bg-warning">Timing</span>
                                                    @break
                                                @case('keywords')
                                                    <span class="badge bg-info">Keywords</span>
                                                    @break
                                                @case('patterns')
                                                    <span class="badge bg-secondary">Patterns</span>
                                                    @break
                                                @case('ip_reputation')
                                                    <span class="badge bg-dark">IP Reputation</span>
                                                    @break
                                                @default
                                                    <span class="badge bg-light text-dark">{{ ucfirst($type) }}</span>
                                            @endswitch
                                        </td>
                                        <td>{{ $count }}</td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <p class="text-muted">No spam attempts recorded in the last {{ $days }} days.</p>
                    @endif
                </div>
            </div>
        </div>

        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5>Top Spam IPs (Last {{ $days }} days)</h5>
                </div>
                <div class="card-body">
                    @if(count($topSpamIps) > 0)
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>IP Address</th>
                                        <th>Attempts</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($topSpamIps as $ip)
                                    <tr>
                                        <td><code>{{ $ip->ip_address }}</code></td>
                                        <td>{{ $ip->attempt_count }}</td>
                                        <td>
                                            <button class="btn btn-sm btn-danger block-ip" data-ip="{{ $ip->ip_address }}">
                                                Block IP
                                            </button>
                                        </td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <p class="text-muted">No spam IPs recorded in the last {{ $days }} days.</p>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <!-- Recent Attempts -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5>Recent Spam Attempts</h5>
                    <div>
                        <button class="btn btn-sm btn-outline-danger" id="clear-old-attempts">
                            Clear Old Attempts
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    @if(count($recentAttempts) > 0)
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Date/Time</th>
                                        <th>IP Address</th>
                                        <th>Form Type</th>
                                        <th>Spam Type</th>
                                        <th>Details</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($recentAttempts as $attempt)
                                    <tr>
                                        <td>{{ $attempt->created_at->format('Y-m-d H:i:s') }}</td>
                                        <td><code>{{ $attempt->ip_address }}</code></td>
                                        <td>
                                            @if($attempt->form_type === 'contact')
                                                <span class="badge bg-primary">Contact</span>
                                            @else
                                                <span class="badge bg-warning">Career</span>
                                            @endif
                                        </td>
                                        <td>
                                            @switch($attempt->spam_type)
                                                @case('honeypot')
                                                    <span class="badge bg-danger">Honeypot</span>
                                                    @break
                                                @case('timing')
                                                    <span class="badge bg-warning">Timing</span>
                                                    @break
                                                @case('keywords')
                                                    <span class="badge bg-info">Keywords</span>
                                                    @break
                                                @case('patterns')
                                                    <span class="badge bg-secondary">Patterns</span>
                                                    @break
                                                @case('ip_reputation')
                                                    <span class="badge bg-dark">IP Reputation</span>
                                                    @break
                                                @default
                                                    <span class="badge bg-light text-dark">{{ ucfirst($attempt->spam_type) }}</span>
                                            @endswitch
                                        </td>
                                        <td>
                                            <small class="text-muted">{{ Str::limit($attempt->details, 50) }}</small>
                                        </td>
                                        <td>
                                            @if($attempt->blocked)
                                                <span class="badge bg-success">Blocked</span>
                                            @else
                                                <span class="badge bg-secondary">Detected</span>
                                            @endif
                                        </td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <p class="text-muted">No recent spam attempts found.</p>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Block IP functionality
    document.querySelectorAll('.block-ip').forEach(button => {
        button.addEventListener('click', function() {
            const ip = this.dataset.ip;
            if (confirm(`Are you sure you want to block IP ${ip}?`)) {
                fetch('{{ route("admin.spam.block-ip") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ ip: ip })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert(data.message);
                        location.reload();
                    } else {
                        alert('Error: ' + data.error);
                    }
                });
            }
        });
    });

    // Clear old attempts
    document.getElementById('clear-old-attempts').addEventListener('click', function() {
        if (confirm('Are you sure you want to clear old spam attempts? This action cannot be undone.')) {
            fetch('{{ route("admin.spam.clear-old") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ days: 30 })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert(data.message);
                    location.reload();
                } else {
                    alert('Error: ' + data.error);
                }
            });
        }
    });
});
</script>
@endsection
