# SEO Implementation Guide for Phoenix Beverages Group

## Overview

This Laravel project now includes a comprehensive SEO system that handles meta tags, Open Graph, Twitter Cards, structured data, and sitemap generation without external dependencies.

## Features Implemented

### ✅ Meta Tags
- Title tags
- Meta descriptions
- Keywords
- Author information
- Robots directives
- Canonical URLs

### ✅ Social Media Optimization
- Open Graph tags (Facebook, LinkedIn)
- Twitter Card tags
- Social media images

### ✅ Structured Data (JSON-LD)
- Organization schema
- Breadcrumb schema
- Custom JSON-LD support

### ✅ Sitemap Generation
- XML sitemap generation
- Robots.txt generation
- Dynamic URL inclusion
- Priority and change frequency settings

## How to Use

### 1. In Controllers

Add SEO data to your controller methods:

```php
public function yourPage()
{
    $seo = [
        'title' => 'Your Page Title - Phoenix Beverages Group',
        'description' => 'Your page description here...',
        'keywords' => 'keyword1, keyword2, keyword3',
        'image' => asset('images/your-image.png'),
        'url' => request()->url(),
        'type' => 'website', // or 'article', 'product', etc.
        'canonical' => request()->url()
    ];
    
    return view('your-view', [
        'seo' => $seo,
        // other data...
    ]);
}
```

### 2. In Views

The SEO meta tags are automatically rendered in the `<head>` section when you pass `$seo` data to your views.

### 3. Generate Sitemap

Run the command to generate sitemap and robots.txt:

```bash
php artisan seo:generate-sitemap
```

### 4. Access Sitemap

- Sitemap: `https://yourdomain.com/sitemap.xml`
- Robots.txt: `https://yourdomain.com/robots.txt`

## SEO Service Methods

### Basic SEO Setup
```php
$seoService = app('App\Services\SeoService');

$seoService->setTitle('Your Title')
           ->setDescription('Your description')
           ->setKeywords('keyword1, keyword2')
           ->setImage('image-url')
           ->setUrl('page-url')
           ->setType('website')
           ->setAuthor('Author Name')
           ->setCanonical('canonical-url')
           ->setRobots('index, follow');
```

### Generate Meta Tags
```php
echo $seoService->render();
```

### Generate Structured Data
```php
// Organization schema
echo $seoService->generateOrganizationSchema();

// Breadcrumb schema
$breadcrumbs = [
    ['name' => 'Home', 'url' => '/'],
    ['name' => 'About', 'url' => '/about']
];
echo $seoService->generateBreadcrumbSchema($breadcrumbs);

// Custom JSON-LD
$customData = [
    '@context' => 'https://schema.org',
    '@type' => 'Article',
    'headline' => 'Article Title'
];
echo $seoService->renderJsonLd($customData);
```

## Sitemap Service

### Add URLs to Sitemap
```php
$sitemapService = new \App\Services\SitemapService();

$sitemapService->addUrl('/page', 0.8, 'weekly')
               ->addUrl('/another-page', 0.9, 'daily');
```

### Generate Sitemap
```php
$xml = $sitemapService->generate();
$sitemapService->save(); // Saves to public/sitemap.xml
```

## SEO Best Practices

### 1. Page Titles
- Keep under 60 characters
- Include brand name
- Be descriptive and unique

### 2. Meta Descriptions
- Keep under 160 characters
- Include primary keywords
- Include call-to-action when appropriate

### 3. Keywords
- Use relevant, specific keywords
- Don't overstuff
- Include long-tail keywords

### 4. Images
- Use descriptive alt text
- Optimize image sizes
- Use appropriate formats (JPG, PNG, WebP)

### 5. URLs
- Use descriptive, keyword-rich URLs
- Keep URLs short and clean
- Use hyphens to separate words

## Testing SEO

### 1. View Page Source
Check that meta tags are properly generated.

### 2. Google Search Console
Submit your sitemap and monitor indexing.

### 3. Social Media Testing
- Facebook: https://developers.facebook.com/tools/debug/
- Twitter: https://cards-dev.twitter.com/validator/
- LinkedIn: https://www.linkedin.com/post-inspector/

### 4. Structured Data Testing
- Google: https://search.google.com/test/rich-results
- Schema.org: https://validator.schema.org/

## Scheduled Sitemap Generation

Add to your `app/Console/Kernel.php`:

```php
protected function schedule(Schedule $schedule)
{
    $schedule->command('seo:generate-sitemap')->daily();
}
```

## Customization

### Adding New SEO Features

1. Extend the `SeoService` class
2. Add new methods for your specific needs
3. Update the Blade directive if needed

### Custom Structured Data

Create custom JSON-LD schemas for:
- Products
- Events
- Reviews
- Local Business
- And more...

## Troubleshooting

### Common Issues

1. **Meta tags not showing**: Ensure `$seo` is passed to view
2. **Sitemap errors**: Check model relationships and database
3. **Structured data errors**: Validate JSON-LD syntax

### Debug Commands

```bash
# Test sitemap generation
php artisan seo:generate-sitemap

# Check sitemap validity
curl https://yourdomain.com/sitemap.xml

# Test robots.txt
curl https://yourdomain.com/robots.txt
```

## Performance Tips

1. **Cache SEO data** for frequently accessed pages
2. **Optimize images** before using in meta tags
3. **Use CDN** for faster image loading
4. **Minimize HTTP requests** by combining CSS/JS

## Security Considerations

1. **Sanitize user input** in SEO data
2. **Validate URLs** before adding to sitemap
3. **Use HTTPS** for all canonical URLs
4. **Protect admin routes** from indexing

---

This SEO implementation provides a solid foundation for search engine optimization while maintaining flexibility for future enhancements. 