<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SeoDataResource\Pages;
use App\Models\SeoData;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\KeyValue;
use Filament\Forms\Components\Repeater;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;

class SeoDataResource extends Resource
{
    protected static ?string $model = SeoData::class;

    protected static ?string $navigationIcon = 'heroicon-o-magnifying-glass';

    protected static ?string $navigationLabel = 'SEO Management';

    protected static ?string $modelLabel = 'SEO Data';

    protected static ?string $pluralModelLabel = 'SEO Data';

    protected static ?string $navigationGroup = 'Content Management';

    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Tabs::make('SEO Configuration')
                    ->tabs([
                        Tabs\Tab::make('Basic SEO')
                            ->schema([
                                Section::make('Page Information')
                                    ->schema([
                                        TextInput::make('page_route')
                                            ->label('Page Route')
                                            ->required()
                                            ->unique(ignoreRecord: true)
                                            ->placeholder('e.g., home, about-us, news-room')
                                            ->helperText('The route identifier for this page (no slashes)'),
                                        
                                        TextInput::make('page_name')
                                            ->label('Page Name')
                                            ->required()
                                            ->placeholder('e.g., Homepage, About Us, News Room')
                                            ->helperText('Human-readable name for this page'),
                                        
                                        Toggle::make('is_active')
                                            ->label('Active')
                                            ->default(true)
                                            ->helperText('Enable/disable SEO for this page'),
                                    ])
                                    ->columns(2),

                                Section::make('Meta Tags')
                                    ->schema([
                                        TextInput::make('title')
                                            ->label('Page Title')
                                            ->required()
                                            ->maxLength(60)
                                            ->helperText('Keep under 60 characters for optimal SEO'),
                                        
                                        Textarea::make('description')
                                            ->label('Meta Description')
                                            ->required()
                                            ->rows(3)
                                            ->maxLength(160)
                                            ->helperText('Keep under 160 characters for optimal SEO'),
                                        
                                        Textarea::make('keywords')
                                            ->label('Keywords')
                                            ->rows(2)
                                            ->placeholder('keyword1, keyword2, keyword3')
                                            ->helperText('Comma-separated keywords for this page'),
                                        
                                        TextInput::make('author')
                                            ->label('Author')
                                            ->placeholder('Phoenix Beverages Group'),
                                        
                                        Select::make('robots')
                                            ->label('Robots Directive')
                                            ->options([
                                                'index, follow' => 'Index, Follow',
                                                'noindex, follow' => 'No Index, Follow',
                                                'index, nofollow' => 'Index, No Follow',
                                                'noindex, nofollow' => 'No Index, No Follow',
                                            ])
                                            ->default('index, follow'),
                                    ])
                                    ->columns(2),
                            ]),

                        Tabs\Tab::make('Advanced SEO')
                            ->schema([
                                Section::make('URLs & Canonical')
                                    ->schema([
                                        TextInput::make('url')
                                            ->label('Page URL')
                                            ->url()
                                            ->placeholder('https://yourdomain.com/page')
                                            ->helperText('Full URL for this page'),
                                        
                                        TextInput::make('canonical')
                                            ->label('Canonical URL')
                                            ->url()
                                            ->placeholder('https://yourdomain.com/page')
                                            ->helperText('Canonical URL to prevent duplicate content'),
                                        
                                        Select::make('type')
                                            ->label('Content Type')
                                            ->options([
                                                'website' => 'Website',
                                                'article' => 'Article',
                                                'product' => 'Product',
                                                'profile' => 'Profile',
                                                'video' => 'Video',
                                                'music' => 'Music',
                                            ])
                                            ->default('website'),
                                    ])
                                    ->columns(2),

                                Section::make('Social Media Images')
                                    ->schema([
                                        FileUpload::make('image')
                                            ->label('Social Media Image')
                                            ->image()
                                            ->directory('seo-images')
                                            ->visibility('public')
                                            ->acceptedFileTypes(['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'])
                                            ->maxSize(5120) // 5MB max
                                            ->imageResizeMode('cover')
                                            ->imageCropAspectRatio('16:9')
                                            ->imageResizeTargetWidth('1200')
                                            ->imageResizeTargetHeight('630')
                                            ->helperText('Image for social media sharing (1200x630px recommended). Accepted formats: JPG, PNG, GIF, WebP. Max size: 2MB'),
                                    ]),
                            ]),

                        Tabs\Tab::make('Open Graph')
                            ->schema([
                                Section::make('Open Graph Tags')
                                    ->schema([
                                        KeyValue::make('open_graph')
                                            ->label('Additional Open Graph Data')
                                            ->keyLabel('Property')
                                            ->valueLabel('Content')
                                            ->addActionLabel('Add OG Property')
                                            ->helperText('Add custom Open Graph properties (e.g., og:site_name, og:locale)'),
                                    ]),
                            ]),

                        Tabs\Tab::make('Twitter Cards')
                            ->schema([
                                Section::make('Twitter Card Data')
                                    ->schema([
                                        KeyValue::make('twitter_card')
                                            ->label('Additional Twitter Card Data')
                                            ->keyLabel('Property')
                                            ->valueLabel('Content')
                                            ->addActionLabel('Add Twitter Property')
                                            ->helperText('Add custom Twitter Card properties (e.g., twitter:site, twitter:creator)'),
                                    ]),
                            ]),

                        Tabs\Tab::make('Structured Data')
                            ->schema([
                                Section::make('JSON-LD Structured Data')
                                    ->schema([
                                        Repeater::make('structured_data')
                                            ->label('Structured Data Blocks')
                                            ->schema([
                                                Select::make('type')
                                                    ->label('Schema Type')
                                                    ->options([
                                                        'Organization' => 'Organization',
                                                        'Article' => 'Article',
                                                        'Product' => 'Product',
                                                        'BreadcrumbList' => 'Breadcrumb List',
                                                        'WebSite' => 'Website',
                                                        'LocalBusiness' => 'Local Business',
                                                    ])
                                                    ->required(),
                                                
                                                KeyValue::make('data')
                                                    ->label('Schema Data')
                                                    ->keyLabel('Property')
                                                    ->valueLabel('Value')
                                                    ->addActionLabel('Add Property'),
                                            ])
                                            ->columns(2)
                                            ->addActionLabel('Add Structured Data Block')
                                            ->helperText('Add JSON-LD structured data for better search engine understanding'),
                                    ]),
                            ]),

                        Tabs\Tab::make('International SEO')
                            ->schema([
                                Section::make('Hreflang & Alternate URLs')
                                    ->schema([
                                        KeyValue::make('hreflang')
                                            ->label('Hreflang Tags')
                                            ->keyLabel('Language Code (e.g., en, fr, es)')
                                            ->valueLabel('URL')
                                            ->helperText('Add hreflang tags for international SEO. Example: en -> https://example.com/en/page'),
                                        
                                        Repeater::make('alternate_urls')
                                            ->label('Alternate URLs')
                                            ->schema([
                                                TextInput::make('lang')
                                                    ->label('Language Code')
                                                    ->required()
                                                    ->placeholder('en')
                                                    ->helperText('Language code (e.g., en, fr, es)'),
                                                TextInput::make('url')
                                                    ->label('URL')
                                                    ->required()
                                                    ->url()
                                                    ->placeholder('https://example.com/en/page')
                                                    ->helperText('Full URL for this language version'),
                                            ])
                                            ->columns(2)
                                            ->addActionLabel('Add Alternate URL')
                                            ->helperText('Add alternate URLs for different language versions of this page'),
                                    ]),
                            ]),
                    ])
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('page_name')
                    ->label('Page Name')
                    ->searchable()
                    ->sortable(),
                
                TextColumn::make('page_route')
                    ->label('Route')
                    ->searchable()
                    ->sortable()
                    ->badge()
                    ->color('gray'),
                
                TextColumn::make('title')
                    ->label('Title')
                    ->limit(50)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        return strlen($state) > 50 ? $state : null;
                    }),
                
                TextColumn::make('type')
                    ->label('Type')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'website' => 'success',
                        'article' => 'info',
                        'product' => 'warning',
                        default => 'gray',
                    }),
                
                IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                
                TextColumn::make('updated_at')
                    ->label('Last Updated')
                    ->dateTime()
                    ->sortable(),
            ])
            ->filters([
                SelectFilter::make('type')
                    ->label('Content Type')
                    ->options([
                        'website' => 'Website',
                        'article' => 'Article',
                        'product' => 'Product',
                        'profile' => 'Profile',
                        'video' => 'Video',
                        'music' => 'Music',
                    ]),
                
                TernaryFilter::make('is_active')
                    ->label('Active Status')
                    ->placeholder('All pages')
                    ->trueLabel('Active only')
                    ->falseLabel('Inactive only'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('page_name');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSeoData::route('/'),
            'create' => Pages\CreateSeoData::route('/create'),
            'edit' => Pages\EditSeoData::route('/{record}/edit'),
        ];
    }

    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::where('is_active', true)->count();
    }
}