<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class News extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'category',
        'tags',
        'image',
        'attachments',
        'body',
        'date',
    ];

    protected $casts = [
        'date' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'attachments' => 'array'
    ];

    private function getAttachmentType($contentType)
    {
        if (str_starts_with($contentType, 'image/')) {
            return 'image';
        } elseif ($contentType === 'application/pdf') {
            return 'pdf';
        } elseif (str_starts_with($contentType, 'video/')) {
            return 'video';
        }
        return 'other';
    }


    public function getDateAttribute($value)
    {
        return Carbon::parse($value)->format('d F, Y');
        // return Cache::remember('news.date.' . $this->id, 3600, function () use ($value) {
           
        // });
    }


    public function scopeOrderByDate($query)
    {
        return $query->orderBy('date', 'desc');
    }


    public function toSearchableArray()
    {
        return [
            'title' => $this->title,
            'description' => $this->description,
            'content' => $this->content,
            'category' => $this->category
        ];
    }



    public function getImageAttribute($value)
    {
        return $value;
    }

    // New accessor for full URL, used in views or API
    public function getImageUrlAttribute()
    {
        $value = $this->attributes['image'] ?? null;

        if (!$value) {
            return null;
        }

        $path = public_path(str_replace('assets/', '', $value));

        if (file_exists($path)) {
            return url(str_replace('assets/', '', $value));
        }

        return url(str_replace('news', 'storage/news/', $value));
    }


    public function getAttachmentsAttribute($value)
    {
        if (empty($value)) {
            return [];
        }

        // Return raw decoded attachments array
        return is_string($value) ? json_decode($value, true) : $value;
    }



    public function getAttachmentsUrlAttribute()
    {
        $attachments = $this->attachments; // raw attachments array

        return array_map(function ($attachment) {
            if (!isset($attachment['url'])) {
                return $attachment;
            }

            $path = public_path(str_replace('assets/', '', $attachment['url']));

            if (file_exists($path)) {
                $attachment['url'] = url(str_replace('assets/', '', $attachment['url']));
            } else {
                $attachment['url'] = url(str_replace('news', 'storage/news/', $attachment['url']));
            }

            return $attachment;
        }, $attachments);
    }

    protected static function booted()
    {
        static::updating(function (News $news) {
            $original = $news->getOriginal('attachments') ?? [];
            $updated = $news->attachments ?? [];

            $deleted = collect($original)
                ->reject(function ($item) use ($updated) {
                    return collect($updated)->contains('url', $item['url']);
                });

            foreach ($deleted as $item) {
                if (!empty($item['url']) && Storage::disk('public')->exists($item['url'])) {
                    Storage::disk('public')->delete($item['url']);
                }
            }
        });
    }
}
