<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class SpamAttempt extends Model
{
    use HasFactory;

    protected $fillable = [
        'ip_address',
        'user_agent',
        'form_type',
        'spam_type',
        'details',
        'blocked',
        'blocked_at'
    ];

    protected $casts = [
        'blocked' => 'boolean',
        'blocked_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Get recent spam attempts for an IP
     */
    public static function getRecentAttemptsByIp(string $ip, int $hours = 24): int
    {
        return static::where('ip_address', $ip)
            ->where('created_at', '>=', now()->subHours($hours))
            ->count();
    }

    /**
     * Get spam statistics
     */
    public static function getSpamStats(int $days = 7): array
    {
        $startDate = now()->subDays($days);
        
        return [
            'total_attempts' => static::where('created_at', '>=', $startDate)->count(),
            'blocked_attempts' => static::where('created_at', '>=', $startDate)
                ->where('blocked', true)
                ->count(),
            'by_type' => static::where('created_at', '>=', $startDate)
                ->selectRaw('spam_type, COUNT(*) as count')
                ->groupBy('spam_type')
                ->pluck('count', 'spam_type')
                ->toArray(),
            'by_form' => static::where('created_at', '>=', $startDate)
                ->selectRaw('form_type, COUNT(*) as count')
                ->groupBy('form_type')
                ->pluck('count', 'form_type')
                ->toArray(),
        ];
    }
}
