/**
 * PDF Renderer for 3D Flipbook
 * Converts PDF pages to images for 3D flipbook display
 */

class PDFRenderer {
    constructor() {
        this.pdfDoc = null;
        this.canvas = null;
        this.ctx = null;
        this.currentPage = 1;
        this.totalPages = 0;
        
        // DearFlip-style: Dynamic DPI rendering for performance
        this.devicePixelRatio = window.devicePixelRatio || 1;
        this.baseScale = 1.5; // Base scale for performance
        this.maxScale = 3.0; // Maximum scale for quality
        this.scale = Math.min(this.baseScale * this.devicePixelRatio, this.maxScale);
        
        // Dynamic resolution balancing (DearFlip-style)
        this.performanceMode = this.devicePixelRatio > 2 ? 'high' : 'balanced';
        this.targetDPI = this.performanceMode === 'high' ? 150 : 200; // Lower DPI for performance
        
        // Adaptive scaling for text sharpness
        this.isFlipping = false;
        this.textSharpnessMode = 'high'; // 'high', 'balanced', 'performance'
        this.multiResolutionCache = new Map(); // Cache at multiple resolutions
        this.textLayerEnabled = true; // Enable vector text layer overlay
        this.currentZoom = 1.0; // Track current zoom level
        this.onPageUpscaled = null; // Callback for when page is upscaled
        
        // Enhanced caching system with separate caches for different data types
        this.renderedPages = new Map(); // In-memory cache for rendered canvases (high quality)
        this.lowQualityPages = new Map(); // Cache for low-quality preview versions
        this.qualityUpgrades = new Set(); // Track pages currently being upgraded
        this.flipbookToPdfMapping = new Map(); // Cache flipbook page -> PDF page mapping
        this.gpuReadyImages = new Map(); // GPU-ready decoded images cache
        this.textLayerCache = new Map(); // Text layer data cache
        
        // Fullscreen optimization
        this.isFullscreen = false;
        this.checkFullscreenMode();
        document.addEventListener('fullscreenchange', () => this.checkFullscreenMode());
        document.addEventListener('webkitfullscreenchange', () => this.checkFullscreenMode());
        document.addEventListener('mozfullscreenchange', () => this.checkFullscreenMode());
        document.addEventListener('MSFullscreenChange', () => this.checkFullscreenMode());
        this.cacheKey = null; // Unique cache key for this PDF
        this.maxCacheSize = 50; // Maximum pages to keep in memory
        this.cacheOrder = []; // LRU order tracking
        this.storagePrefix = 'pdf_cache_';
        this.maxStorageSize = 100 * 1024 * 1024; // 100MB storage limit
        
        // Separate cache limits for different data types
        this.maxRenderedPages = 30; // Rendered canvas cache limit
        this.maxGPUImages = 20; // GPU-ready images cache limit
        this.maxTextLayers = 15; // Text layer cache limit
        
        // Predictive preloading with rolling buffer
        this.preloadQueue = [];
        this.isPreloading = false;
        this.preloadRadius = 4; // Increased to ±4 pages for better performance
        this.preloadBuffer = new Set(); // Track preloaded pages
        
        // GPU-ready image management
        this.gpuReadyImages = new Map(); // Images ready for GPU compositing
        this.decodePromises = new Map(); // Track decode operations
        
        // Performance monitoring
        this.cacheHits = 0;
        this.cacheMisses = 0;
        this.renderTimes = [];
        this.preloadTimes = [];
        
        // Memory management
        this.canvasPool = []; // Reuse canvas elements
        this.maxCanvasPoolSize = 10;
        
        // Web Worker for PDF rendering (DearFlip-style)
        this.worker = null;
        this.workerTasks = new Map();
        this.workerTaskId = 0;
        
        // Active cache management (DearFlip-style: keep only 4-6 spreads)
        this.activePages = new Set();
        this.maxActivePages = 12; // 6 spreads = 12 pages
        this.currentSpread = 1;
    }

    /**
     * Check if browser is in fullscreen mode
     */
    checkFullscreenMode() {
        this.isFullscreen = !!(
            document.fullscreenElement ||
            document.webkitFullscreenElement ||
            document.mozFullScreenElement ||
            document.msFullscreenElement
        );
    }

    /**
     * Get optimized container dimensions based on screen size and fullscreen mode
     */
    getContainerDimensions() {
        const isFullscreen = this.isFullscreen;
        if (isFullscreen) {
            // Fullscreen: use maximum available space for maximum clarity
            return {
                width: window.innerWidth * 0.55, // 55% width for bigger pages
                height: window.innerHeight * 0.95, // 95% height for maximum use
                maxScale: 3.5, // Higher scale cap for fullscreen
                qualityMultiplier: 1.5 // 150% resolution in fullscreen
            };
        } else {
            // Normal mode
            return {
                width: window.innerWidth * 0.50,
                height: window.innerHeight * 0.90,
                maxScale: 2.5,
                qualityMultiplier: 1.3
            };
        }
        
        // Initialize Web Worker
        this.initializeWorker();
    }

    /**
     * Check if browser is in fullscreen mode
     */
    checkFullscreenMode() {
        this.isFullscreen = !!(
            document.fullscreenElement ||
            document.webkitFullscreenElement ||
            document.mozFullScreenElement ||
            document.msFullscreenElement
        );
    }

    /**
     * Get optimized container dimensions based on screen size and fullscreen mode
     */
    getContainerDimensions() {
        const isFullscreen = this.isFullscreen;
        if (isFullscreen) {
            // Fullscreen: use maximum available space for maximum clarity
            return {
                width: window.innerWidth * 0.55, // 55% width for bigger pages
                height: window.innerHeight * 0.95, // 95% height for maximum use
                maxScale: 3.5, // Higher scale cap for fullscreen
                qualityMultiplier: 1.5 // 150% resolution in fullscreen
            };
        } else {
            // Normal mode
            return {
                width: window.innerWidth * 0.50,
                height: window.innerHeight * 0.90,
                maxScale: 2.5,
                qualityMultiplier: 1.3
            };
        }
    }

    /**
     * Get proper render scale including devicePixelRatio and zoom level
     */
    getRenderScale(isFlipping = false) {
        const baseScale = this.getAdaptiveScale(isFlipping);
        const zoomLevel = this.currentZoom || 1.0;
        const dpr = window.devicePixelRatio || 1;
        
        return baseScale * zoomLevel * dpr;
    }

    /**
     * Get adaptive scale based on motion state and device capabilities
     */
    getAdaptiveScale(isFlipping = false) {
        if (isFlipping) {
            // Lower scale during motion for smooth animation
            return this.baseScale * 0.75;
        }
        
        const dpr = window.devicePixelRatio || 1;
        const visualBoost = dpr >= 2 ? 1.25 : 1.0;
        
        // Apply text sharpness mode
        let sharpnessMultiplier = 1.0;
        switch (this.textSharpnessMode) {
            case 'high':
                sharpnessMultiplier = 1.5;
                break;
            case 'balanced':
                sharpnessMultiplier = 1.2;
                break;
            case 'performance':
                sharpnessMultiplier = 1.0;
                break;
        }
        
        return Math.min(
            this.baseScale * dpr * visualBoost * sharpnessMultiplier, 
            this.maxScale
        );
    }

    /**
     * Set text sharpness mode
     */
    setTextSharpnessMode(mode) {
        this.textSharpnessMode = mode;
        console.log(`Text sharpness mode set to: ${mode}`);
    }

    /**
     * Set flipping state for adaptive scaling
     */
    setFlippingState(isFlipping) {
        this.isFlipping = isFlipping;
    }

    /**
     * Enhanced multi-threaded render pipeline with parallel processing
     */
    initializeWorker() {
        if (typeof Worker !== 'undefined' && typeof OffscreenCanvas !== 'undefined') {
            try {
                // Create inline Web Worker for PDF rendering with OffscreenCanvas
                const workerCode = `
                    // Enhanced PDF rendering worker with parallel processing
                    self.onmessage = async function(e) {
                        const { taskId, type, data } = e.data;
                        
                        if (type === 'renderPage') {
                            try {
                                const { pageNumber, scale, pdfData, viewport } = data;
                                
                                // Create OffscreenCanvas with proper dimensions
                                const canvas = new OffscreenCanvas(viewport.width, viewport.height);
                                const ctx = canvas.getContext('2d', {
                                    alpha: false,
                                    desynchronized: true
                                });
                                
                                // Professional text rendering settings
                                ctx.imageSmoothingEnabled = false;
                                ctx.textRenderingOptimization = 'geometricPrecision';
                                ctx.filter = 'contrast(110%) brightness(105%)';
                                
                                // White background
                                ctx.fillStyle = 'white';
                                ctx.fillRect(0, 0, canvas.width, canvas.height);
                                
                                // Simulate PDF rendering (in real implementation, would use PDF.js)
                                ctx.fillStyle = 'black';
                                ctx.font = '16px Arial';
                                ctx.fillText('Page ' + pageNumber, 50, 50);
                                
                                // Convert to ImageBitmap for GPU efficiency
                                const bitmap = canvas.transferToImageBitmap();
                                
                                // Convert to blob for transfer (PNG for better text clarity)
                                const blob = await canvas.convertToBlob({ 
                                    type: 'image/png'
                                });
                                
                                self.postMessage({
                                    taskId: taskId,
                                    type: 'renderComplete',
                                    data: { pageNumber, blob, bitmap }
                                }, [bitmap]);
                            } catch (error) {
                                self.postMessage({
                                    taskId: taskId,
                                    type: 'renderError',
                                    data: error.message
                                });
                            }
                        }
                        
                        if (type === 'batchRender') {
                            try {
                                const { pages, scale } = data;
                                const results = [];
                                
                                // Parallel processing of multiple pages
                                const promises = pages.map(async (pageData) => {
                                    const canvas = new OffscreenCanvas(pageData.width, pageData.height);
                                    const ctx = canvas.getContext('2d', {
                                        alpha: false,
                                        desynchronized: true
                                    });
                                    
                                    // Professional settings
                                    ctx.imageSmoothingEnabled = false;
                                    ctx.textRenderingOptimization = 'geometricPrecision';
                                    
                                    // Render page
                                    ctx.fillStyle = 'white';
                                    ctx.fillRect(0, 0, canvas.width, canvas.height);
                                    ctx.fillStyle = 'black';
                                    ctx.font = '16px Arial';
                                    ctx.fillText('Page ' + pageData.pageNumber, 50, 50);
                                    
                                    const blob = await canvas.convertToBlob({ 
                                        type: 'image/png'
                                    });
                                    
                                    return {
                                        pageNumber: pageData.pageNumber,
                                        blob: blob,
                                        width: canvas.width,
                                        height: canvas.height
                                    };
                                });
                                
                                const batchResults = await Promise.all(promises);
                                
                                self.postMessage({
                                    taskId: taskId,
                                    type: 'batchComplete',
                                    data: batchResults
                                });
                            } catch (error) {
                                self.postMessage({
                                    taskId: taskId,
                                    type: 'batchError',
                                    data: error.message
                                });
                            }
                        }
                    };
                `;
                
                const blob = new Blob([workerCode], { type: 'application/javascript' });
                this.worker = new Worker(URL.createObjectURL(blob));
                
                this.worker.onmessage = (e) => {
                    const { taskId, type, data } = e.data;
                    const task = this.workerTasks.get(taskId);
                    
                    if (task) {
                        if (type === 'renderComplete') {
                            task.resolve(data);
                        } else if (type === 'batchComplete') {
                            task.resolve(data);
                        } else if (type === 'renderError' || type === 'batchError') {
                            task.reject(new Error(data));
                        }
                        this.workerTasks.delete(taskId);
                    }
                };
                
                this.worker.onerror = (error) => {
                    console.error('Web Worker error:', error);
                };
                
                console.log('Enhanced multi-threaded render pipeline initialized');
            } catch (error) {
                console.warn('Web Worker not supported, falling back to main thread:', error);
                this.worker = null;
            }
        } else {
            console.warn('OffscreenCanvas not supported, using main thread rendering');
            this.worker = null;
        }
    }

    /**
     * DearFlip-style aggressive cache trimming
     * Keep only 4-6 spreads (8-12 pages) in memory
     */
    trimCache(keepAlivePages = new Set()) {
        const pagesToKeep = new Set(keepAlivePages);
        
        // Add current spread ±2 spreads
        const currentPage = this.currentSpread * 2 - 1;
        for (let i = Math.max(1, currentPage - 4); i <= Math.min(this.totalPages, currentPage + 4); i++) {
            pagesToKeep.add(i);
        }
        
        // Remove pages not in keep-alive set
        const pagesToRemove = [];
        for (const [pageNum, pageData] of this.renderedPages) {
            if (!pagesToKeep.has(pageNum)) {
                pagesToRemove.push(pageNum);
            }
        }
        
        // Clean up removed pages
        pagesToRemove.forEach(pageNum => {
            this.renderedPages.delete(pageNum);
            this.gpuReadyImages.delete(pageNum);
            this.cacheOrder = this.cacheOrder.filter(p => p !== pageNum);
        });
        
        // Update active pages
        this.activePages = pagesToKeep;
        
        console.log(`Cache trimmed: removed ${pagesToRemove.length} pages, keeping ${this.renderedPages.size} pages`);
    }

    /**
     * OffscreenCanvas rendering for maximum performance and text sharpness
     */
    async renderWithOffscreenCanvas(page, viewport) {
        if (typeof OffscreenCanvas === 'undefined') {
            throw new Error('OffscreenCanvas not supported');
        }
        
        try {
            const offscreen = new OffscreenCanvas(viewport.width, viewport.height);
            const offCtx = offscreen.getContext('2d', { 
                alpha: false,
                desynchronized: true
            });
            
            // Professional text rendering settings
            offCtx.imageSmoothingEnabled = false;
            offCtx.textRenderingOptimization = 'geometricPrecision';
            offCtx.filter = 'contrast(110%) brightness(105%)';
            
            // White background
            offCtx.fillStyle = 'white';
            offCtx.fillRect(0, 0, offscreen.width, offscreen.height);
            
            // Render with print intent for crisp text
            const renderContext = {
                canvasContext: offCtx,
                viewport: viewport,
                intent: 'print',
                renderInteractiveForms: true,
                renderAnnotations: true
            };
            
            await page.render(renderContext).promise;
            
            // Convert to ImageBitmap for GPU efficiency
            const bitmap = offscreen.transferToImageBitmap();
            
            // Convert to blob for storage
            const blob = await bitmapToBlob(bitmap);
            const dataURL = URL.createObjectURL(blob);
            
            return {
                dataURL: dataURL,
                width: offscreen.width,
                height: offscreen.height,
                bitmap: bitmap,
                isGPUReady: true
            };
        } catch (error) {
            console.warn('OffscreenCanvas rendering failed, falling back to main thread:', error);
            throw error;
        }
    }

    /**
     * Pre-decode image for GPU texture upload (DearFlip-style)
     */
    async preDecodeImage(img) {
        if (img.decode) {
            try {
                await img.decode();
                return true;
            } catch (error) {
                console.warn('Image decode failed:', error);
                return false;
            }
        }
        return true; // Fallback for browsers without decode support
    }

    /**
     * Get or create canvas from pool for GPU optimization
     */
    getCanvasFromPool(width, height) {
        // Try to reuse existing canvas
        for (let i = 0; i < this.canvasPool.length; i++) {
            const canvas = this.canvasPool[i];
            if (canvas.width >= width && canvas.height >= height) {
                this.canvasPool.splice(i, 1);
                canvas.width = width;
                canvas.height = height;
                return canvas;
            }
        }
        
        // Create new canvas with GPU optimization
        const canvas = document.createElement('canvas');
        canvas.width = width;
        canvas.height = height;
        
        // Enable GPU acceleration with text sharpness optimization
        const ctx = canvas.getContext('2d', { 
            willReadFrequently: false, // Optimize for GPU
            alpha: false, // Disable alpha for better performance
            desynchronized: true // Allow async rendering
        });
        
        // Professional text rendering settings for maximum sharpness
        ctx.imageSmoothingEnabled = false; // Disable for sharper text edges
        ctx.imageSmoothingQuality = 'high'; // High quality when smoothing is needed
        ctx.textRenderingOptimization = 'geometricPrecision'; // Sharpest text rendering
        ctx.textBaseline = 'alphabetic';
        ctx.textAlign = 'left';
        
        // Remove filters that may blur text - let text be naturally sharp
        // ctx.filter = 'contrast(110%) brightness(105%)'; // Commented out to prevent text blur
        
        return canvas;
    }

    /**
     * Return canvas to pool for reuse
     */
    returnCanvasToPool(canvas) {
        if (this.canvasPool.length < this.maxCanvasPoolSize) {
            // Clear the canvas
            const ctx = canvas.getContext('2d');
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            this.canvasPool.push(canvas);
        }
    }

    /**
     * Load PDF document with enhanced performance monitoring
     */
    async loadPDF(pdfUrl) {
        const startTime = performance.now();
        
        try {
            console.log('Loading PDF:', pdfUrl);
            console.log(`Device pixel ratio: ${this.devicePixelRatio}, Scale: ${this.scale}`);
            
            // Generate cache key based on PDF URL and scale
            this.cacheKey = this.generateCacheKey(pdfUrl);
            console.log('Cache key:', this.cacheKey);
            
            // Load PDF.js if not already loaded
            if (typeof pdfjsLib === 'undefined') {
                await this.loadPDFJS();
            }

            // Load the PDF document
            const loadingTask = pdfjsLib.getDocument(pdfUrl);
            this.pdfDoc = await loadingTask.promise;
            this.totalPages = this.pdfDoc.numPages;
            
            // Load cached pages from storage
            await this.loadCachedPages();
            
            const loadTime = performance.now() - startTime;
            console.log(`PDF loaded: ${this.totalPages} pages in ${loadTime.toFixed(2)}ms`);
            console.log(`Cached pages loaded: ${this.renderedPages.size}`);
            console.log(`GPU-ready images: ${this.gpuReadyImages.size}`);
            
            return this.pdfDoc;
        } catch (error) {
            console.error('Error loading PDF:', error);
            throw error;
        }
    }

    /**
     * Load PDF.js library
     */
    async loadPDFJS() {
        return new Promise((resolve, reject) => {
            if (typeof pdfjsLib !== 'undefined') {
                resolve();
                return;
            }

            const script = document.createElement('script');
            script.src = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/4.0.379/pdf.min.js';
            script.onload = () => {
                // Configure PDF.js
                pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/4.0.379/pdf.worker.min.js';
                
                // Restore original console.warn after PDF.js loads
                console.warn = originalWarn;
                
                resolve();
            };
            script.onerror = reject;
            document.head.appendChild(script);
        });
    }

    /**
     * Generate unique cache key for PDF
     */
    generateCacheKey(pdfUrl) {
        // Create a hash from PDF URL and scale for cache key
        const urlHash = btoa(pdfUrl).replace(/[^a-zA-Z0-9]/g, '');
        return `${this.storagePrefix}${urlHash}_scale${this.scale}`;
    }

    /**
     * Load cached pages from browser storage
     */
    async loadCachedPages() {
        if (!this.cacheKey) return;
        
        try {
            const cachedData = localStorage.getItem(this.cacheKey);
            if (cachedData) {
                const parsed = JSON.parse(cachedData);
                const { pages, timestamp } = parsed;
                
                // Check if cache is not too old (7 days)
                const cacheAge = Date.now() - timestamp;
                const maxAge = 7 * 24 * 60 * 60 * 1000; // 7 days
                
                if (cacheAge < maxAge) {
                    // Restore cached pages
                    pages.forEach(pageData => {
                        this.renderedPages.set(pageData.pageNumber, pageData);
                        this.updateCacheOrder(pageData.pageNumber);
                    });
                    console.log(`Loaded ${pages.length} cached pages from storage`);
                } else {
                    console.log('Cache expired, clearing old cache');
                    localStorage.removeItem(this.cacheKey);
                }
            }
        } catch (error) {
            console.warn('Error loading cached pages:', error);
            // Clear corrupted cache
            if (this.cacheKey) {
                localStorage.removeItem(this.cacheKey);
            }
        }
    }

    /**
     * Save pages to browser storage
     */
    async saveCachedPages() {
        if (!this.cacheKey || this.renderedPages.size === 0) return;
        
        try {
            const pages = Array.from(this.renderedPages.values());
            const cacheData = {
                pages: pages,
                timestamp: Date.now(),
                scale: this.scale
            };
            
            const dataString = JSON.stringify(cacheData);
            
            // Check storage size limit
            if (dataString.length > this.maxStorageSize) {
                console.log('Cache too large, removing oldest pages');
                this.cleanupOldCache();
                return this.saveCachedPages(); // Retry with smaller cache
            }
            
            localStorage.setItem(this.cacheKey, dataString);
            console.log(`Saved ${pages.length} pages to cache`);
        } catch (error) {
            console.warn('Error saving cached pages:', error);
            // If storage is full, try to clean up
            if (error.name === 'QuotaExceededError') {
                this.cleanupOldCache();
            }
        }
    }

    /**
     * Enhanced LRU cache management with proper ordering
     */
    updateCacheOrder(pageNumber) {
        // Remove from current position if exists
        const index = this.cacheOrder.indexOf(pageNumber);
        if (index > -1) {
            this.cacheOrder.splice(index, 1);
        }
        // Add to end (most recently used)
        this.cacheOrder.push(pageNumber);
        
        // Implement LRU eviction if cache is too large
        const maxCacheSize = 10; // Keep more pages for better performance
        if (this.cacheOrder.length > maxCacheSize) {
            const oldestPage = this.cacheOrder.shift();
            this.renderedPages.delete(oldestPage);
            this.gpuReadyImages.delete(oldestPage);
            this.textLayerCache.delete(oldestPage);
            console.log(`LRU evicted page ${oldestPage}`);
        }
    }

    /**
     * Clean up old cache entries
     */
    cleanupOldCache() {
        // Remove oldest 25% of cached pages
        const removeCount = Math.floor(this.renderedPages.size * 0.25);
        for (let i = 0; i < removeCount && this.cacheOrder.length > 0; i++) {
            const oldestPage = this.cacheOrder.shift();
            this.renderedPages.delete(oldestPage);
        }
        console.log(`Cleaned up ${removeCount} old cache entries`);
    }

    /**
     * Predictive preloader — warms cache and GPU early
     */
    async preloadPages(pageNumbers) {
        // Ensure pageNumbers is an array
        const pageNumbersArray = Array.isArray(pageNumbers) ? pageNumbers : [pageNumbers];
        
        for (const p of pageNumbersArray) {
            if (!this.renderedPages.has(p) && p > 0 && p <= this.totalPages) {
                const imgData = await this.getPageAsImage(p, 1.0);
                const img = new Image();
                img.decoding = 'async';
                img.loading = 'eager';
                img.src = imgData.dataURL;
                await img.decode().catch(() => {});
            }
        }
    }

    /**
     * Preload pages using Web Worker for maximum performance
     */
    async preloadPagesWithWorker(pageNumbers) {
        const taskId = ++this.workerTaskId;
        
        // Prepare page data for worker
        const pagesData = pageNumbers.map(pageNum => ({
            pageNumber: pageNum,
            width: 800, // Will be calculated properly in real implementation
            height: 1000
        }));
        
        return new Promise((resolve, reject) => {
            this.workerTasks.set(taskId, { resolve, reject });
            
            this.worker.postMessage({
                taskId: taskId,
                type: 'batchRender',
                data: { pages: pagesData, scale: this.getAdaptiveScale(false) }
            });
        });
    }

    /**
     * Progressive enhancement: Fast preview first, then high-res in background
     * Makes page flips feel instant even with heavy PDFs
     */
    async getPageAsImageProgressive(pageNumber, callback = null) {
        // Step 1: Render fast preview first (0.75x scale)
        const preview = await this.getPageAsImage(pageNumber, 0.75);
        console.log(`Fast preview rendered for page ${pageNumber}`);
        
        // Call callback with preview immediately
        if (callback) {
            callback(preview);
        }
        
        // Step 2: Schedule high-res render in background
        if (window.requestIdleCallback) {
            requestIdleCallback(async () => {
                try {
                    const highRes = await this.getPageAsImage(pageNumber, 1.5);
                    console.log(`High-res render completed for page ${pageNumber}`);
                    
                    // Update cache with high-res version
                    this.renderedPages.set(pageNumber, highRes);
                    
                    // Trigger UI update
                    this.onPageUpscaled?.(pageNumber, highRes);
                } catch (error) {
                    console.warn(`Error upscaling page ${pageNumber}:`, error);
                }
            });
        } else {
            // Fallback for browsers without requestIdleCallback
            setTimeout(async () => {
                try {
                    const highRes = await this.getPageAsImage(pageNumber, 1.5);
                    this.renderedPages.set(pageNumber, highRes);
                    this.onPageUpscaled?.(pageNumber, highRes);
                } catch (error) {
                    console.warn(`Error upscaling page ${pageNumber}:`, error);
                }
            }, 100);
        }
        
        return preview;
    }

    /**
     * Progressive rendering: Fast low-res first, then high-res when idle
     */
    async renderPageWithProgressiveUpscaling(pageNumber) {
        // Step 1: Render fast low-res preview for instant display
        const lowResData = await this.renderPageFixed(pageNumber, { scale: 1.0 });
        console.log(`Low-res preview rendered for page ${pageNumber}`);
        
        // Step 2: Schedule high-res render when idle
        if (window.requestIdleCallback) {
            requestIdleCallback(async () => {
                try {
                    const highResData = await this.renderPageFixed(pageNumber, { scale: 2.0 });
                    console.log(`High-res render completed for page ${pageNumber}`);
                    
                    // Replace low-res with high-res in cache
                    this.renderedPages.set(pageNumber, highResData);
                    
                    // Trigger UI update if needed
                    this.onPageUpscaled?.(pageNumber, highResData);
        } catch (error) {
                    console.warn(`Error upscaling page ${pageNumber}:`, error);
                }
            });
        } else {
            // Fallback for browsers without requestIdleCallback
            setTimeout(async () => {
                try {
                    const highResData = await this.renderPageFixed(pageNumber, { scale: 2.0 });
                    this.renderedPages.set(pageNumber, highResData);
                    this.onPageUpscaled?.(pageNumber, highResData);
                } catch (error) {
                    console.warn(`Error upscaling page ${pageNumber}:`, error);
                }
            }, 100);
        }
        
        return lowResData;
    }

    /**
     * Render pages concurrently with proper batching
     */
    async renderPagesConcurrently(pageNumbers) {
        const concurrencyLimit = 3; // Process 3 pages at a time
        const results = [];
        
        for (let i = 0; i < pageNumbers.length; i += concurrencyLimit) {
            const batch = pageNumbers.slice(i, i + concurrencyLimit);
            const batchResults = await Promise.all(
                batch.map(pageNum => this.renderPageFixed(pageNum))
            );
            results.push(...batchResults);
        }
        
        return results;
    }

    /**
     * Parallel preloading on main thread with requestIdleCallback
     */
    async preloadPagesParallel(pageNumbers) {
        const batchSize = Math.min(4, pageNumbers.length); // Process 4 pages at a time
        const batches = [];
        
        for (let i = 0; i < pageNumbers.length; i += batchSize) {
            batches.push(pageNumbers.slice(i, i + batchSize));
        }
        
        // Process batches with idle time optimization
        for (const batch of batches) {
            await new Promise(resolve => {
                if (window.requestIdleCallback) {
                    requestIdleCallback(async () => {
                        const promises = batch.map(pageNum => this.preloadPageWithGPUOptimization(pageNum));
                        await Promise.all(promises);
                        resolve();
                    });
                } else {
                    // Fallback for browsers without requestIdleCallback
                    setTimeout(async () => {
                        const promises = batch.map(pageNum => this.preloadPageWithGPUOptimization(pageNum));
                        await Promise.all(promises);
                        resolve();
                    }, 0);
                }
            });
        }
    }

    /**
     * Preload single page with GPU optimization
     */
    async preloadPageWithGPUOptimization(pageNumber) {
        try {
            const pageData = await this.getPageAsImage(pageNumber);
            
            // Create GPU-ready image
            const img = new Image();
            img.src = pageData.dataURL;
            
            // Ensure image is GPU-ready
            if (img.decode) {
                await img.decode();
                this.gpuReadyImages.set(pageNumber, img);
                console.log(`Page ${pageNumber} is GPU-ready`);
            }
            
            return pageData;
        } catch (error) {
            console.error(`Error preloading page ${pageNumber}:`, error);
            this.preloadBuffer.delete(pageNumber);
            throw error;
        }
    }

    /**
     * Get GPU-ready image for immediate use
     */
    getGPUReadyImage(pageNumber) {
        return this.gpuReadyImages.get(pageNumber) || null;
    }

    /**
     * Enhanced preload pages around current page (backward compatibility)
     */
    async preloadPagesAround(currentPage) {
        const bufferPages = [];
        
        // Create rolling buffer of ±4 pages
        for (let i = -this.preloadRadius; i <= this.preloadRadius; i++) {
            const pageNum = currentPage + i;
            if (pageNum >= 1 && pageNum <= this.totalPages) {
                bufferPages.push(pageNum);
            }
        }
        
        return this.preloadPages(bufferPages);
    }

    /**
     * Trim cache if it grows too large with texture recycling
     */
    trimCache(limit = 20) {
        if (this.renderedPages.size > limit) {
            const keys = [...this.renderedPages.keys()];
            for (let i = 0; i < this.renderedPages.size - limit; i++) {
                const pageData = this.renderedPages.get(keys[i]);
                if (pageData && pageData.dataURL && pageData.dataURL.startsWith('blob:')) {
                    // Revoke blob URL to free GPU memory
                    URL.revokeObjectURL(pageData.dataURL);
                }
                this.renderedPages.delete(keys[i]);
            }
        }
    }

    /**
     * Enhanced cache management with separate cache types
     */
    manageCacheSize() {
        // Manage rendered pages cache
        if (this.renderedPages.size > this.maxRenderedPages) {
            const excess = this.renderedPages.size - this.maxRenderedPages;
            const pagesToRemove = this.cacheOrder.slice(0, excess);
            pagesToRemove.forEach(pageNum => {
                this.renderedPages.delete(pageNum);
                this.cacheOrder = this.cacheOrder.filter(p => p !== pageNum);
            });
            console.log(`Cleaned up ${excess} rendered pages from cache`);
        }
        
        // Manage GPU-ready images cache
        if (this.gpuReadyImages.size > this.maxGPUImages) {
            const excess = this.gpuReadyImages.size - this.maxGPUImages;
            const gpuKeys = Array.from(this.gpuReadyImages.keys());
            const keysToRemove = gpuKeys.slice(0, excess);
            keysToRemove.forEach(key => this.gpuReadyImages.delete(key));
            console.log(`Cleaned up ${excess} GPU-ready images from cache`);
        }
        
        // Manage text layer cache
        if (this.textLayerCache.size > this.maxTextLayers) {
            const excess = this.textLayerCache.size - this.maxTextLayers;
            const textKeys = Array.from(this.textLayerCache.keys());
            const keysToRemove = textKeys.slice(0, excess);
            keysToRemove.forEach(key => this.textLayerCache.delete(key));
            console.log(`Cleaned up ${excess} text layers from cache`);
        }
    }

    /**
     * Clear all cache with enhanced cleanup
     */
    clearCache() {
        this.renderedPages.clear();
        this.cacheOrder = [];
        this.gpuReadyImages.clear();
        this.textLayerCache.clear();
        this.preloadBuffer.clear();
        this.decodePromises.clear();
        
        // Clear canvas pool
        this.canvasPool.forEach(canvas => {
            const ctx = canvas.getContext('2d');
            ctx.clearRect(0, 0, canvas.width, canvas.height);
        });
        this.canvasPool = [];
        
        // Clear performance metrics
        this.renderTimes = [];
        this.preloadTimes = [];
        this.cacheHits = 0;
        this.cacheMisses = 0;
        
        if (this.cacheKey) {
            localStorage.removeItem(this.cacheKey);
        }
        
        console.log('All cache and GPU resources cleared');
    }

    /**
     * Cleanup method for memory management
     */
    cleanup() {
        this.optimizeMemory();
        this.clearGPUImages();
        
        // Force garbage collection if available
        if (window.gc) {
            window.gc();
        }
        
        console.log('Cleanup completed');
    }

    /**
     * Get enhanced cache statistics with performance metrics
     */
    getCacheStats() {
        const totalSize = Array.from(this.renderedPages.values())
            .reduce((size, page) => size + (page.dataURL ? page.dataURL.length : 0), 0);
        
        const avgRenderTime = this.renderTimes.length > 0 
            ? this.renderTimes.reduce((a, b) => a + b, 0) / this.renderTimes.length 
            : 0;
            
        const avgPreloadTime = this.preloadTimes.length > 0 
            ? this.preloadTimes.reduce((a, b) => a + b, 0) / this.preloadTimes.length 
            : 0;
        
        return {
            cachedPages: this.renderedPages.size,
            totalPages: this.totalPages,
            cacheSize: Math.round(totalSize / 1024) + ' KB',
            cacheHitRate: this.cacheHits / (this.cacheHits + this.cacheMisses) * 100 || 0,
            gpuReadyImages: this.gpuReadyImages.size,
            preloadBuffer: this.preloadBuffer.size,
            avgRenderTime: Math.round(avgRenderTime) + 'ms',
            avgPreloadTime: Math.round(avgPreloadTime) + 'ms',
            canvasPoolSize: this.canvasPool.length,
            devicePixelRatio: this.devicePixelRatio,
            scale: this.scale
        };
    }

    /**
     * Clear GPU-ready images to free memory
     */
    clearGPUImages() {
        this.gpuReadyImages.clear();
        console.log('GPU-ready images cleared');
    }

    /**
     * Optimize memory usage
     */
    optimizeMemory() {
        // Clear old render times (keep last 50)
        if (this.renderTimes.length > 50) {
            this.renderTimes = this.renderTimes.slice(-50);
        }
        
        // Clear old preload times (keep last 50)
        if (this.preloadTimes.length > 50) {
            this.preloadTimes = this.preloadTimes.slice(-50);
        }
        
        // Clean up canvas pool if too large
        while (this.canvasPool.length > this.maxCanvasPoolSize) {
            this.canvasPool.shift();
        }
        
        console.log('Memory optimized');
    }

    /**
     * Get performance metrics
     */
    getPerformanceMetrics() {
        return {
            renderTimes: this.renderTimes,
            preloadTimes: this.preloadTimes,
            cacheHits: this.cacheHits,
            cacheMisses: this.cacheMisses,
            memoryUsage: {
                renderedPages: this.renderedPages.size,
                gpuReadyImages: this.gpuReadyImages.size,
                canvasPool: this.canvasPool.length,
                preloadBuffer: this.preloadBuffer.size
            }
        };
    }

    /**
     * Create text layer overlay for vector text rendering with proper PDF.js integration
     */
    async createTextLayer(page, viewport, container) {
        if (!this.textLayerEnabled) return null;
        
        try {
            const textLayerDiv = document.createElement('div');
            textLayerDiv.classList.add('textLayer');
            textLayerDiv.style.cssText = `
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                pointer-events: none;
                z-index: 10;
                overflow: hidden;
                opacity: 1;
                line-height: 1.0;
                text-align: left;
                color: transparent;
                transform-origin: 0% 0%;
            `;
            
            // Get text content from PDF
            const textContent = await page.getTextContent();
            
            // Create text layer using PDF.js TextLayerBuilder
            const textLayer = new pdfjsLib.TextLayerBuilder({
                textLayerDiv: textLayerDiv,
                pageIndex: page.pageNumber - 1,
                viewport: viewport,
                enhanceTextSelection: true,
                isOffscreenCanvasSupported: typeof OffscreenCanvas !== 'undefined'
            });
            
            // Set text content and render
            textLayer.setTextContent(textContent);
            textLayer.render();
            
            // Apply professional text styling for maximum sharpness
            textLayerDiv.style.cssText += `
                mix-blend-mode: multiply;
                text-shadow: 0 0 1px rgba(0,0,0,0.15);
                font-smooth: always;
                -webkit-font-smoothing: antialiased;
                -moz-osx-font-smoothing: grayscale;
                text-rendering: geometricPrecision;
            `;
            
            // Cache the text layer
            this.textLayerCache.set(page.pageNumber, textLayerDiv);
            
            container.appendChild(textLayerDiv);
            console.log(`Text layer created for page ${page.pageNumber}`);
            return textLayerDiv;
        } catch (error) {
            console.warn('Error creating text layer:', error);
            return null;
        }
    }

    /**
     * Render a specific page to canvas, splitting landscape pages into two portrait pages
     */
    async renderPage(pageNumber, canvas, options = {}) {
        try {
            if (!this.pdfDoc) {
                throw new Error('PDF not loaded');
            }

            const page = await this.pdfDoc.getPage(pageNumber);
            
            // Use proper render scale including DPR and zoom
            const renderScale = this.getRenderScale(this.isFlipping);
            const originalViewport = page.getViewport({ scale: 1.0 });
            
            // Check if this is a landscape page (width > height) that needs splitting
            const isLandscape = originalViewport.width > originalViewport.height;
            
            if (isLandscape) {
                console.log(`Page ${pageNumber} is landscape (${originalViewport.width}x${originalViewport.height}), splitting into two pages`);
                return this.renderSplitPage(pageNumber, page, options);
            } else {
                // Regular portrait page
                return this.renderPortraitPage(pageNumber, page, options);
            }
        } catch (error) {
            console.error(`Error rendering page ${pageNumber}:`, error);
            throw error;
        }
    }

    /**
     * Render a landscape page split into two portrait pages with GPU optimization
     */
    async renderSplitPage(pageNumber, page, options = {}) {
        const startTime = performance.now();
        const originalViewport = page.getViewport({ scale: 1.0 });
        
        // Get optimized container dimensions (adjusts for fullscreen mode)
        const containerDims = this.getContainerDimensions();
        const pageContainerWidth = containerDims.width;
        const pageContainerHeight = containerDims.height;
        
        // Calculate scale so that full landscape width fits 2x the page container width
        const widthScale = (pageContainerWidth * 2) / originalViewport.width;
        const heightScale = pageContainerHeight / originalViewport.height;
        const baseScale = Math.min(widthScale, heightScale, containerDims.maxScale); // Dynamic cap based on mode
        
        // Step 3: Progressive quality - low quality for speed, high quality for maximum clarity
        const isLowQuality = options.quality === 'low';
        // Use higher quality multiplier in fullscreen for maximum clarity
        const qualityMultiplier = isLowQuality ? 0.65 : containerDims.qualityMultiplier;
        
        // Display size stays constant, but render resolution differs
        const dpr = window.devicePixelRatio || 1;
        // For display, use baseScale. For render resolution, apply quality multiplier
        const displayScale = baseScale * dpr; // Target display size
        const renderScale = displayScale * qualityMultiplier; // Actual render resolution (higher for clarity)
        const finalScale = renderScale; // Render at this scale, will be upscaled for display if low quality
        
        const viewport = page.getViewport({ scale: finalScale });
        
        // Use canvas pool for better memory management
        const tempCanvas = this.getCanvasFromPool(viewport.width, viewport.height);
        const tempCtx = tempCanvas.getContext('2d', { 
            willReadFrequently: false, // GPU optimization
            alpha: false,
            desynchronized: true
        });
        
        // Rendering settings: high quality = sharp text, low quality = smooth upscaling
        if (isLowQuality) {
            tempCtx.imageSmoothingEnabled = true;
            tempCtx.imageSmoothingQuality = 'high';
            tempCtx.textRenderingOptimization = 'optimizeQuality';
        } else {
            // High quality: maximum clarity and sharpness
            tempCtx.imageSmoothingEnabled = false; // Disable for sharper text edges
            tempCtx.textRenderingOptimization = 'geometricPrecision'; // Sharpest text rendering
            // Enhanced contrast for better text visibility (more aggressive in fullscreen)
            const contrastBoost = this.isFullscreen ? 1.12 : 1.08;
            const brightnessBoost = this.isFullscreen ? 1.04 : 1.02;
            tempCtx.filter = `contrast(${contrastBoost}) brightness(${brightnessBoost})`;
        }
        
        // Pure white background (#FFFFFF) for maximum contrast
        tempCtx.fillStyle = '#FFFFFF';
        tempCtx.fillRect(0, 0, tempCanvas.width, tempCanvas.height);
        
        // Render the full landscape page with professional text settings
        const renderContext = {
            canvasContext: tempCtx,
            viewport: viewport,
            intent: isLowQuality ? 'display' : 'print', // Print intent for crisp text in high quality
            renderInteractiveForms: true,
            renderAnnotations: true,
            // Additional options for maximum quality
            enableWebGL: false, // Use 2D canvas for better text rendering
            transform: null
        };
        await page.render(renderContext).promise;
        
        // Determine which half to extract
        const isLeftHalf = options.half === 'left';
        const halfWidth = Math.floor(viewport.width / 2);
        
        // Create final canvas with GPU optimization
        const finalCanvas = this.getCanvasFromPool(halfWidth, viewport.height);
        const finalCtx = finalCanvas.getContext('2d', { 
            willReadFrequently: false,
            alpha: false,
            desynchronized: true
        });
        
        // Rendering settings: high quality = sharp text, low quality = smooth upscaling
        if (isLowQuality) {
            finalCtx.imageSmoothingEnabled = true;
            finalCtx.imageSmoothingQuality = 'high';
            finalCtx.textRenderingOptimization = 'optimizeQuality';
        } else {
            // High quality: maximum clarity and sharpness
            finalCtx.imageSmoothingEnabled = false; // Disable for sharper text edges
            finalCtx.textRenderingOptimization = 'geometricPrecision'; // Sharpest text rendering
            // Enhanced contrast for better text visibility (more aggressive in fullscreen)
            const contrastBoost = this.isFullscreen ? 1.12 : 1.08;
            const brightnessBoost = this.isFullscreen ? 1.04 : 1.02;
            finalCtx.filter = `contrast(${contrastBoost}) brightness(${brightnessBoost})`;
        }
        
        // Pure white background (#FFFFFF) for maximum contrast
        finalCtx.fillStyle = '#FFFFFF';
        finalCtx.fillRect(0, 0, finalCanvas.width, finalCanvas.height);
        
        // Extract the appropriate half with GPU-optimized drawImage
        const sourceX = isLeftHalf ? 0 : halfWidth;
        finalCtx.drawImage(
            tempCanvas,
            sourceX, 0, halfWidth, viewport.height,
            0, 0, halfWidth, viewport.height
        );
        
        // If low quality, upscale to display size for consistent appearance
        if (isLowQuality) {
            const displayHalfWidth = Math.floor((pageContainerWidth * dpr));
            const displayHeight = Math.floor((pageContainerHeight * dpr));
            const upscaledCanvas = this.getCanvasFromPool(displayHalfWidth, displayHeight);
            const upscaledCtx = upscaledCanvas.getContext('2d', {
                willReadFrequently: false,
                alpha: false,
                desynchronized: true
            });
            
            // Use high-quality image smoothing for upscaling with slight sharpening
            upscaledCtx.imageSmoothingEnabled = true;
            upscaledCtx.imageSmoothingQuality = 'high';
            // Subtle sharpening filter for better upscaled text
            upscaledCtx.filter = 'contrast(1.06) brightness(1.01)';
            
            // Pure white background for maximum contrast
            upscaledCtx.fillStyle = '#FFFFFF';
            upscaledCtx.fillRect(0, 0, upscaledCanvas.width, upscaledCanvas.height);
            
            // Upscale the low-quality render to display size
            upscaledCtx.drawImage(finalCanvas, 0, 0, finalCanvas.width, finalCanvas.height, 
                                  0, 0, upscaledCanvas.width, upscaledCanvas.height);
            
            // Return canvases to pool
            this.returnCanvasToPool(tempCanvas);
            this.returnCanvasToPool(finalCanvas);
            
            const renderTime = performance.now() - startTime;
            this.renderTimes.push(renderTime);
            
            console.log(`Page ${pageNumber} split into ${isLeftHalf ? 'left' : 'right'} half (LOW QUALITY upscaled) in ${renderTime.toFixed(2)}ms, dimensions: ${upscaledCanvas.width}x${upscaledCanvas.height}`);
            return upscaledCanvas;
        }
        
        // Return temp canvas to pool
        this.returnCanvasToPool(tempCanvas);
        
        const renderTime = performance.now() - startTime;
        this.renderTimes.push(renderTime);
        
        console.log(`Page ${pageNumber} split into ${isLeftHalf ? 'left' : 'right'} half (HIGH QUALITY) in ${renderTime.toFixed(2)}ms, dimensions: ${finalCanvas.width}x${finalCanvas.height}`);
        return finalCanvas;
    }

    /**
     * Render a regular portrait page with GPU optimization
     */
    async renderPortraitPage(pageNumber, page, options = {}) {
        const startTime = performance.now();
        const originalViewport = page.getViewport({ scale: 1.0 });
        
        // Get optimized container dimensions (adjusts for fullscreen mode)
        const containerDims = this.getContainerDimensions();
        const pageContainerWidth = containerDims.width;
        const pageContainerHeight = containerDims.height;
        
        // Calculate scale to fit within page container
        const widthScale = pageContainerWidth / originalViewport.width;
        const heightScale = pageContainerHeight / originalViewport.height;
        const baseScale = Math.min(widthScale, heightScale, containerDims.maxScale); // Dynamic cap based on mode
        
        // Step 3: Progressive quality - low quality for speed, high quality for maximum clarity
        const isLowQuality = options.quality === 'low';
        // Use higher quality multiplier in fullscreen for maximum clarity
        const qualityMultiplier = isLowQuality ? 0.65 : containerDims.qualityMultiplier;
        
        // Display size stays constant, but render resolution differs
        const dpr = window.devicePixelRatio || 1;
        // For display, use baseScale. For render resolution, apply quality multiplier
        const displayScale = baseScale * dpr; // Target display size
        const renderScale = displayScale * qualityMultiplier; // Actual render resolution (higher for clarity)
        const finalScale = renderScale; // Render at this scale, will be upscaled for display if low quality
        
        const viewport = page.getViewport({ scale: finalScale });
        
        // Use canvas pool for better memory management
        const canvas = this.getCanvasFromPool(viewport.width, viewport.height);
        const ctx = canvas.getContext('2d', { 
            willReadFrequently: false, // GPU optimization
            alpha: false,
            desynchronized: true
        });
        
        // Rendering settings: high quality = sharp text, low quality = smooth upscaling
        if (isLowQuality) {
            ctx.imageSmoothingEnabled = true;
            ctx.imageSmoothingQuality = 'high';
            ctx.textRenderingOptimization = 'optimizeQuality';
        } else {
            // High quality: maximum clarity and sharpness
            ctx.imageSmoothingEnabled = false; // Disable for sharper text edges
            ctx.textRenderingOptimization = 'geometricPrecision'; // Sharpest text rendering
            // Enhanced contrast for better text visibility (more aggressive in fullscreen)
            const contrastBoost = this.isFullscreen ? 1.12 : 1.08;
            const brightnessBoost = this.isFullscreen ? 1.04 : 1.02;
            ctx.filter = `contrast(${contrastBoost}) brightness(${brightnessBoost})`;
        }
        
        // Pure white background (#FFFFFF) for maximum contrast
        ctx.fillStyle = '#FFFFFF';
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        
        // Render the page with professional text settings
        const renderContext = {
            canvasContext: ctx,
            viewport: viewport,
            intent: isLowQuality ? 'display' : 'print', // Print intent for crisp text in high quality
            renderInteractiveForms: true,
            renderAnnotations: true,
            // Additional options for maximum quality
            enableWebGL: false, // Use 2D canvas for better text rendering
            transform: null
        };
        await page.render(renderContext).promise;
        
        // If low quality, upscale to display size for consistent appearance
        if (isLowQuality) {
            const displayWidth = Math.floor((pageContainerWidth * dpr));
            const displayHeight = Math.floor((pageContainerHeight * dpr));
            const upscaledCanvas = this.getCanvasFromPool(displayWidth, displayHeight);
            const upscaledCtx = upscaledCanvas.getContext('2d', {
                willReadFrequently: false,
                alpha: false,
                desynchronized: true
            });
            
            // Use high-quality image smoothing for upscaling with slight sharpening
            upscaledCtx.imageSmoothingEnabled = true;
            upscaledCtx.imageSmoothingQuality = 'high';
            // Subtle sharpening filter for better upscaled text
            upscaledCtx.filter = 'contrast(1.06) brightness(1.01)';
            
            // Pure white background for maximum contrast
            upscaledCtx.fillStyle = '#FFFFFF';
            upscaledCtx.fillRect(0, 0, upscaledCanvas.width, upscaledCanvas.height);
            
            // Upscale the low-quality render to display size
            upscaledCtx.drawImage(canvas, 0, 0, canvas.width, canvas.height, 
                                  0, 0, upscaledCanvas.width, upscaledCanvas.height);
            
            // Return original canvas to pool
            this.returnCanvasToPool(canvas);
            
            const renderTime = performance.now() - startTime;
            this.renderTimes.push(renderTime);
            
            console.log(`Page ${pageNumber} rendered as portrait (LOW QUALITY upscaled) in ${renderTime.toFixed(2)}ms, dimensions: ${upscaledCanvas.width}x${upscaledCanvas.height}`);
            return upscaledCanvas;
        }
        
        const renderTime = performance.now() - startTime;
        this.renderTimes.push(renderTime);
        
        console.log(`Page ${pageNumber} rendered as portrait (HIGH QUALITY) in ${renderTime.toFixed(2)}ms, dimensions: ${canvas.width}x${canvas.height}`);
        return canvas;
    }

    /**
     * Helper: Convert flipbook page number to PDF page number and half
     * This accounts for landscape pages being split into two flipbook pages
     * Uses caching for performance
     */
    async getPdfPageInfoForFlipbookPage(flipbookPageNumber) {
        // Check cache first
        if (this.flipbookToPdfMapping.has(flipbookPageNumber)) {
            return this.flipbookToPdfMapping.get(flipbookPageNumber);
        }
        
        let currentFlipbookPage = 1;
        
        // Scan through PDF pages to find which PDF page corresponds to this flipbook page
        for (let pdfPageNum = 1; pdfPageNum <= this.totalPages; pdfPageNum++) {
            const page = await this.pdfDoc.getPage(pdfPageNum);
            const viewport = page.getViewport({ scale: 1.0 });
            const isLandscape = viewport.width > viewport.height;
            
            if (isLandscape) {
                // Landscape page = 2 flipbook pages
                if (currentFlipbookPage === flipbookPageNumber) {
                    const info = { pdfPage: pdfPageNum, half: 'left' };
                    this.flipbookToPdfMapping.set(flipbookPageNumber, info);
                    return info;
                }
                currentFlipbookPage++;
                if (currentFlipbookPage === flipbookPageNumber) {
                    const info = { pdfPage: pdfPageNum, half: 'right' };
                    this.flipbookToPdfMapping.set(flipbookPageNumber, info);
                    return info;
                }
                currentFlipbookPage++;
            } else {
                // Portrait page = 1 flipbook page
                if (currentFlipbookPage === flipbookPageNumber) {
                    const info = { pdfPage: pdfPageNum, half: null };
                    this.flipbookToPdfMapping.set(flipbookPageNumber, info);
                    return info;
                }
                currentFlipbookPage++;
            }
            
            // Stop if we've passed the target flipbook page
            if (currentFlipbookPage > flipbookPageNumber) {
                break;
            }
        }
        
        return null; // Should not happen
    }

    /**
     * Render page at low quality for fast display during flips
     * Step 1: Low-quality rendering method - handles flipbook page numbers
     */
    async renderPageLowQuality(flipbookPageNumber) {
        // Check cache first
        if (this.lowQualityPages.has(flipbookPageNumber)) {
            const cached = this.lowQualityPages.get(flipbookPageNumber);
            console.log(`Using cached low-quality flipbook page ${flipbookPageNumber}`);
            return cached;
        }

        try {
            console.log(`Rendering flipbook page ${flipbookPageNumber} at low quality for fast display...`);
            
            // Get PDF page info for this flipbook page number
            const pageInfo = await this.getPdfPageInfoForFlipbookPage(flipbookPageNumber);
            if (!pageInfo) {
                throw new Error(`Could not find PDF page info for flipbook page ${flipbookPageNumber}`);
            }
            
            // Render at low quality (50% scale) with correct half if landscape
            const renderOptions = { 
                quality: 'low',
                half: pageInfo.half // 'left', 'right', or null
            };
            const canvas = await this.renderPage(pageInfo.pdfPage, null, renderOptions);
            
            // Convert to JPEG at 70% quality for smaller file size
            const dataURL = canvas.toDataURL('image/jpeg', 0.7);
            
            const pageData = {
                pageNumber: flipbookPageNumber,
                dataURL,
                width: canvas.width,
                height: canvas.height,
                quality: 'low',
                originalPdfPage: pageInfo.pdfPage,
                half: pageInfo.half,
                timestamp: Date.now()
            };

            // Cache the low-quality version
            this.lowQualityPages.set(flipbookPageNumber, pageData);
            
            console.log(`Flipbook page ${flipbookPageNumber} (PDF ${pageInfo.pdfPage}${pageInfo.half ? ' ' + pageInfo.half : ''}) rendered at low quality: ${canvas.width}x${canvas.height}`);
            return pageData;
        } catch (error) {
            console.error(`Error rendering low-quality flipbook page ${flipbookPageNumber}:`, error);
            throw error;
        }
    }

    /**
     * Upgrade page from low quality to high quality
     * Step 2: High-quality upgrade method - handles flipbook page numbers
     */
    async upgradePageQuality(flipbookPageNumber) {
        // Check if already upgrading or already have high quality
        if (this.qualityUpgrades.has(flipbookPageNumber)) {
            console.log(`Flipbook page ${flipbookPageNumber} is already being upgraded`);
            return null; // Already upgrading
        }

        if (this.renderedPages.has(flipbookPageNumber)) {
            // Already have high quality
            const cached = this.renderedPages.get(flipbookPageNumber);
            console.log(`Flipbook page ${flipbookPageNumber} already at high quality`);
            return cached;
        }

        // Get the low quality version to extract PDF page info
        const lowQualityData = this.lowQualityPages.get(flipbookPageNumber);
        if (!lowQualityData) {
            console.log(`No low-quality version found for flipbook page ${flipbookPageNumber}, cannot upgrade`);
            return null;
        }

        // Mark as upgrading to prevent duplicate upgrades
        this.qualityUpgrades.add(flipbookPageNumber);

        try {
            console.log(`Upgrading flipbook page ${flipbookPageNumber} to high quality...`);
            
            // Use the PDF page info from low quality data
            const renderOptions = {
                quality: 'high',
                half: lowQualityData.half // 'left', 'right', or null
            };
            
            // Render at full quality
            const canvas = await this.renderPage(lowQualityData.originalPdfPage, null, renderOptions);
            
            // Convert to PNG for highest quality
            const dataURL = canvas.toDataURL('image/png', 1.0);
            
            const pageData = {
                pageNumber: flipbookPageNumber,
                dataURL,
                width: canvas.width,
                height: canvas.height,
                quality: 'high',
                originalPdfPage: lowQualityData.originalPdfPage,
                half: lowQualityData.half,
                timestamp: Date.now()
            };

            // Store in high quality cache
            this.renderedPages.set(flipbookPageNumber, pageData);
            
            console.log(`Flipbook page ${flipbookPageNumber} upgraded to high quality: ${canvas.width}x${canvas.height}`);
            return pageData;
        } catch (error) {
            console.error(`Error upgrading flipbook page ${flipbookPageNumber}:`, error);
            throw error;
        } finally {
            // Always remove from upgrading set
            this.qualityUpgrades.delete(flipbookPageNumber);
        }
    }

    /**
     * Render the first visible pages (for initial view) with proper landscape splitting
     */
    async renderInitialPages(count = 4) {
        const results = [];
        let flipbookPageNumber = 1;
        
        for (let i = 1; i <= Math.min(count, this.totalPages); i++) {
            const page = await this.pdfDoc.getPage(i);
            const originalViewport = page.getViewport({ scale: 1.0 });
            const isLandscape = originalViewport.width > originalViewport.height;
            
            if (isLandscape) {
                // Split landscape page into two pages
                const leftCanvas = await this.renderPage(i, null, { half: 'left' });
                const leftDataURL = leftCanvas.toDataURL('image/png', 1.0);
                results.push({
                    pageNumber: flipbookPageNumber++,
                    dataURL: leftDataURL,
                    width: leftCanvas.width,
                    height: leftCanvas.height,
                    originalPdfPage: i,
                    half: 'left'
                });
                
                const rightCanvas = await this.renderPage(i, null, { half: 'right' });
                const rightDataURL = rightCanvas.toDataURL('image/png', 1.0);
                results.push({
                    pageNumber: flipbookPageNumber++,
                    dataURL: rightDataURL,
                    width: rightCanvas.width,
                    height: rightCanvas.height,
                    originalPdfPage: i,
                    half: 'right'
                });
            } else {
                // Regular portrait page
                const canvas = await this.renderPage(i, null, {});
                const dataURL = canvas.toDataURL('image/png', 1.0);
                results.push({
                    pageNumber: flipbookPageNumber++,
                    dataURL: dataURL,
                    width: canvas.width,
                    height: canvas.height,
                    originalPdfPage: i,
                    half: null
                });
            }
        }
        
        return results;
    }

    /**
     * Progressive background rendering for remaining pages with landscape splitting
     */
    async renderRemainingPages(startPage, onProgress) {
        let flipbookPageNumber = startPage;
        
        for (let i = startPage; i <= this.totalPages; i++) {
            const page = await this.pdfDoc.getPage(i);
            const originalViewport = page.getViewport({ scale: 1.0 });
            const isLandscape = originalViewport.width > originalViewport.height;
            
            if (isLandscape) {
                // Split landscape page into two pages
                const leftCanvas = await this.renderPage(i, null, { half: 'left' });
                const leftDataURL = leftCanvas.toDataURL('image/png', 1.0);
                const leftPage = {
                    pageNumber: flipbookPageNumber++,
                    dataURL: leftDataURL,
                    width: leftCanvas.width,
                    height: leftCanvas.height,
                    originalPdfPage: i,
                    half: 'left'
                };
                onProgress?.(leftPage);
                
                const rightCanvas = await this.renderPage(i, null, { half: 'right' });
                const rightDataURL = rightCanvas.toDataURL('image/png', 1.0);
                const rightPage = {
                    pageNumber: flipbookPageNumber++,
                    dataURL: rightDataURL,
                    width: rightCanvas.width,
                    height: rightCanvas.height,
                    originalPdfPage: i,
                    half: 'right'
                };
                onProgress?.(rightPage);
            } else {
                // Regular portrait page
                const canvas = await this.renderPage(i, null, {});
                const dataURL = canvas.toDataURL('image/png', 1.0);
                const pageData = {
                    pageNumber: flipbookPageNumber++,
                    dataURL: dataURL,
                    width: canvas.width,
                    height: canvas.height,
                    originalPdfPage: i,
                    half: null
                };
                onProgress?.(pageData);
            }
            
            await new Promise(r => requestIdleCallback(r)); // yield for UI
        }
    }

    /**
     * Render all pages and return as image data URLs, splitting landscape pages
     * @deprecated Use renderInitialPages() and renderRemainingPages() for better performance
     */
    async renderAllPages() {
        const pages = [];
        let flipbookPageNumber = 1;
        
        for (let i = 1; i <= this.totalPages; i++) {
            const page = await this.pdfDoc.getPage(i);
            const originalViewport = page.getViewport({ scale: 1.0 });
            const isLandscape = originalViewport.width > originalViewport.height;
            
            if (isLandscape) {
                // Split landscape page into two pages
                // Left half
                const leftCanvas = await this.renderPage(i, null, { half: 'left' });
                const leftDataURL = leftCanvas.toDataURL('image/png', 1.0);
                pages.push({
                    pageNumber: flipbookPageNumber++,
                    dataURL: leftDataURL,
                    width: leftCanvas.width,
                    height: leftCanvas.height,
                    originalPdfPage: i,
                    half: 'left'
                });
                
                // Right half
                const rightCanvas = await this.renderPage(i, null, { half: 'right' });
                const rightDataURL = rightCanvas.toDataURL('image/png', 1.0);
                pages.push({
                    pageNumber: flipbookPageNumber++,
                    dataURL: rightDataURL,
                    width: rightCanvas.width,
                    height: rightCanvas.height,
                    originalPdfPage: i,
                    half: 'right'
                });
            } else {
                // Regular portrait page
                const canvas = await this.renderPage(i, null, {});
                const dataURL = canvas.toDataURL('image/png', 1.0);
                pages.push({
                    pageNumber: flipbookPageNumber++,
                    dataURL: dataURL,
                    width: canvas.width,
                    height: canvas.height,
                    originalPdfPage: i,
                    half: null
                });
            }
        }
        
        return pages;
    }

    /**
     * Multi-resolution caching for crisp text and smooth animation
     */
    async getPageWithMultiResolution(flipbookPageNumber, isFlipping = false) {
        const cacheKey = `${flipbookPageNumber}_${isFlipping ? 'low' : 'high'}`;
        
        // Check multi-resolution cache first
        if (this.multiResolutionCache.has(cacheKey)) {
            console.log(`Multi-res cache HIT for ${cacheKey}`);
            return this.multiResolutionCache.get(cacheKey);
        }
        
        // Render at appropriate resolution
        const scale = isFlipping ? this.getAdaptiveScale(true) : this.getAdaptiveScale(false);
        const pageData = await this.renderPageAtScale(flipbookPageNumber, scale);
        
        // Cache the result
        this.multiResolutionCache.set(cacheKey, pageData);
        
        // Clean up old cache entries
        if (this.multiResolutionCache.size > 20) {
            const firstKey = this.multiResolutionCache.keys().next().value;
            this.multiResolutionCache.delete(firstKey);
        }
        
        return pageData;
    }

    /**
     * Render page at specific scale
     */
    async renderPageAtScale(pageNumber, scale) {
        const page = await this.pdfDoc.getPage(pageNumber);
        const originalViewport = page.getViewport({ scale: 1.0 });
        const targetHeight = window.innerHeight * 0.9;
        
        const heightScale = (targetHeight * scale) / originalViewport.height;
        const viewport = page.getViewport({ scale: heightScale });
        
        const canvas = this.getCanvasFromPool(viewport.width, viewport.height);
        const ctx = canvas.getContext('2d', { 
            willReadFrequently: false,
            alpha: false,
            desynchronized: true
        });
        
        // Professional text rendering settings
        ctx.imageSmoothingEnabled = false;
        ctx.textRenderingOptimization = 'geometricPrecision';
        // Remove filters that may blur text - let text be naturally sharp
        // ctx.filter = 'contrast(110%) brightness(105%)'; // Commented out to prevent text blur
        
        // White background
        ctx.fillStyle = 'white';
        ctx.fillRect(0, 0, canvas.width, canvas.height);
        
        // Render with print intent for crisp text
        const renderContext = {
            canvasContext: ctx,
            viewport: viewport,
            intent: 'print',
            renderInteractiveForms: true,
            renderAnnotations: true
        };
        
        await page.render(renderContext).promise;
        
                const dataURL = canvas.toDataURL('image/png', 1.0);
        this.returnCanvasToPool(canvas);
        
        return {
            pageNumber: pageNumber,
                    dataURL: dataURL,
                    width: canvas.width,
                    height: canvas.height,
            scale: scale,
            isGPUReady: false
        };
    }

    /**
     * Fixed page rendering with proper canvas sizing and devicePixelRatio scaling
     */
    async renderPageFixed(pageNumber, options = {}) {
        const startTime = performance.now();
        
        if (!this.pdfDoc) {
            throw new Error('PDF not loaded');
        }

        const page = await this.pdfDoc.getPage(pageNumber);
        
        // FIXED: Use both base scale and device pixel ratio for crisp rendering
        const baseScale = options.scale || 1.5; // Default 1.5x base scale
        const finalScale = baseScale * window.devicePixelRatio;
        const viewport = page.getViewport({ scale: finalScale });

        // Create canvas with full internal resolution
        const canvas = this.getCanvasFromPool(viewport.width, viewport.height);
        const ctx = canvas.getContext('2d', { 
            willReadFrequently: true, // Optimize for frequent reads
            alpha: false, 
            desynchronized: true 
        });
        
        // Disable image smoothing for sharp text
        ctx.imageSmoothingEnabled = false;

        // White background
        ctx.fillStyle = 'white';
        ctx.fillRect(0, 0, canvas.width, canvas.height);

        // Render at full resolution
        const renderContext = {
            canvasContext: ctx,
            viewport,
            intent: 'print',
            renderInteractiveForms: true,
            renderAnnotations: true,
        };

        await page.render(renderContext).promise;

        // CRITICAL: Set display size to avoid CSS blur
        canvas.style.width = `${viewport.width / window.devicePixelRatio}px`;
        canvas.style.height = `${viewport.height / window.devicePixelRatio}px`;

        // Convert to data URL for existing pipeline
                const dataURL = canvas.toDataURL('image/png', 1.0);

        const renderTime = performance.now() - startTime;
        console.log(`Page ${pageNumber} rendered with devicePixelRatio scaling in ${renderTime.toFixed(2)}ms`);

        return {
            pageNumber,
            dataURL,
            canvas,
                width: canvas.width,
                    height: canvas.height,
            displayWidth: viewport.width / window.devicePixelRatio,
            displayHeight: viewport.height / window.devicePixelRatio,
            scale: finalScale,
            devicePixelRatio: window.devicePixelRatio,
            isGPUReady: true,
            renderTime
        };
    }

    /**
     * Render a single PDF page with retina resolution and proper sizing
     * Fixes blurriness by using devicePixelRatio scaling
     */
    async getPageAsImage(pageNumber, scale = 1.0) {
        if (!this.pdfDoc) throw new Error('PDF not loaded');
        if (this.renderedPages.has(pageNumber)) return this.renderedPages.get(pageNumber);

        const page = await this.pdfDoc.getPage(pageNumber);
        const originalViewport = page.getViewport({ scale: 1.0 });
        const isLandscape = originalViewport.width > originalViewport.height;
        
        // Get optimized container dimensions (adjusts for fullscreen mode)
        const containerDims = this.getContainerDimensions();
        const pageContainerWidth = containerDims.width;
        const pageContainerHeight = containerDims.height;
        
        let baseScale;
        if (isLandscape) {
            // Landscape pages will be split into two halves
            // Full landscape width should be ~2x the container width
            const fullWidthScale = (pageContainerWidth * 2) / originalViewport.width;
            const heightScale = pageContainerHeight / originalViewport.height;
            baseScale = Math.min(fullWidthScale, heightScale, containerDims.maxScale); // Dynamic cap based on mode
        } else {
            // Portrait pages: fit within single page container
            const widthScale = pageContainerWidth / originalViewport.width;
            const heightScale = pageContainerHeight / originalViewport.height;
            baseScale = Math.min(widthScale, heightScale, containerDims.maxScale); // Dynamic cap based on mode
        }
        
        // Apply DPR for retina resolution
        const dpr = window.devicePixelRatio || 1;
        const finalScale = baseScale * dpr;
        const viewport = page.getViewport({ scale: finalScale });
        
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d', { willReadFrequently: true });

        // INTERNAL pixel size (for rendering quality)
        canvas.width = viewport.width;
        canvas.height = viewport.height;

        // DISPLAY size (for layout) - CRITICAL FIX
        canvas.style.width = `${viewport.width / window.devicePixelRatio}px`;
        canvas.style.height = `${viewport.height / window.devicePixelRatio}px`;

        // Disable image smoothing for sharp text
        ctx.imageSmoothingEnabled = false;

        // White background
        ctx.fillStyle = 'white';
        ctx.fillRect(0, 0, canvas.width, canvas.height);

        // Render the page
        const renderContext = {
            canvasContext: ctx,
            viewport,
            intent: 'print',
            renderInteractiveForms: true,
            renderAnnotations: true,
        };

        await page.render(renderContext).promise;

        // Convert to blob URL for better performance
        const blob = await new Promise((resolve) => canvas.toBlob(resolve, 'image/png'));
        const blobUrl = URL.createObjectURL(blob);

        const pageData = {
            pageNumber,
            dataURL: blobUrl, // Use blob URL instead of data URL
            width: viewport.width / window.devicePixelRatio, // Display width
            height: viewport.height / window.devicePixelRatio, // Display height
            internalWidth: viewport.width, // Internal pixel width
            internalHeight: viewport.height, // Internal pixel height
            originalPdfPage: pageNumber,
            half: null,
            isGPUReady: false,
            scale: finalScale,
            devicePixelRatio: window.devicePixelRatio,
            isLandscape: isLandscape
        };

        this.renderedPages.set(pageNumber, pageData);
        return pageData;
    }

    /**
     * Convert ImageBitmap to Blob
     */
    async bitmapToBlob(bitmap) {
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');
        canvas.width = bitmap.width;
        canvas.height = bitmap.height;
        ctx.drawImage(bitmap, 0, 0);
        
        return new Promise(resolve => {
            canvas.toBlob(resolve, 'image/png', 1.0);
        });
    }

    /**
     * Enhanced image preloading with GPU optimization for flip performance
     */
    async preloadImageWithOptimization(src) {
        const img = new Image();
        img.loading = 'eager'; // Hint to browser preload aggressively
        img.decoding = 'async'; // Use async for better performance
        img.src = src;
        
        // Pre-decode for GPU texture upload
        if (img.decode) {
            await img.decode();
        }
        
        // GPU optimization for flip performance
        img.style.willChange = 'transform';
        
        return img;
    }

    /**
     * GPU warm-up before flip: Ensure pages are uploaded to GPU
     */
    async gpuWarmupBeforeFlip(pageElement) {
        const img = pageElement.querySelector('img');
        if (!img) return;
        
        // GPU optimization for smooth flipping
        img.decoding = 'async';
        img.loading = 'eager';
        img.style.willChange = 'transform, opacity';
        
        // Pre-decode for GPU texture upload
        if (img.decode) {
            await img.decode();
        }
        
        console.log('Page warmed up for GPU-optimized flipping');
    }

    /**
     * Enhanced page ready check with GPU optimization
     * Pre-decodes and pre-caches images before flip
     */
    async ensurePageReady(pageElement) {
        const img = pageElement.querySelector('img');
        if (!img) return;
        
        // Pre-decode for GPU texture upload
        if (!img.complete) {
            await img.decode().catch(() => {});
        }
        
        // Hint GPU to promote this layer before flip
        img.style.willChange = 'transform, opacity';
        img.style.transform = 'translateZ(0)';
        
        console.log('Page ready for GPU-optimized flipping');
    }

    /**
     * Get performance metrics including text sharpness
     */
    getTextSharpnessMetrics() {
        return {
            textSharpnessMode: this.textSharpnessMode,
            adaptiveScale: this.getAdaptiveScale(this.isFlipping),
            multiResolutionCacheSize: this.multiResolutionCache.size,
            textLayerEnabled: this.textLayerEnabled,
            devicePixelRatio: this.devicePixelRatio,
            performanceMode: this.performanceMode
        };
    }

    /**
     * Enable/disable text layer overlay
     */
    setTextLayerEnabled(enabled) {
        this.textLayerEnabled = enabled;
        console.log(`Text layer overlay ${enabled ? 'enabled' : 'disabled'}`);
    }

    /**
     * Handle zoom events for re-rendering at higher scale
     */
    handleZoomEvent(zoomLevel) {
        console.log(`Zoom level changed to: ${zoomLevel}`);
        
        // Update current zoom level
        this.currentZoom = zoomLevel;
        
        // Clear multi-resolution cache to force re-rendering at new scale
        this.multiResolutionCache.clear();
        
        // Clear existing rendered pages to force re-render
        this.renderedPages.clear();
        this.gpuReadyImages.clear();
        this.textLayerCache.clear();
        
        // Re-render current pages at higher scale
        this.rerenderCurrentPages(zoomLevel);
    }

    /**
     * Pre-render on zoom: Re-render visible pages at new zoom level
     * Don't scale existing images — re-render them to avoid blur
     */
    onZoomChange(newScale) {
        console.log(`Zoom change detected: scale ${newScale}`);
        
        // Get currently visible pages (you can customize this logic)
        const visiblePages = Array.from(this.renderedPages.keys()).slice(0, 4); // First 4 pages
        
        for (const pageNum of visiblePages) {
            // Clear existing cache for this page
            const existingData = this.renderedPages.get(pageNum);
            if (existingData && existingData.dataURL && existingData.dataURL.startsWith('blob:')) {
                URL.revokeObjectURL(existingData.dataURL);
            }
            this.renderedPages.delete(pageNum);
            
            // Re-render at new scale
            this.getPageAsImage(pageNum, newScale).then(updated => {
                console.log(`Page ${pageNum} re-rendered at scale ${newScale}`);
                // Trigger UI update
                this.onPageUpscaled?.(pageNum, updated);
            }).catch(error => {
                console.warn(`Error re-rendering page ${pageNum} at scale ${newScale}:`, error);
            });
        }
    }

    /**
     * Lazy re-render on zoom: Re-render visible pages only at new zoom level
     */
    async renderVisiblePagesAtScale(newScale) {
        console.log(`Re-rendering visible pages at scale ${newScale}`);
        
        // Get currently visible pages (you can customize this logic)
        const visiblePages = Array.from(this.renderedPages.keys()).slice(0, 4); // First 4 pages
        
        for (const pageNum of visiblePages) {
            try {
                // Re-render page at new scale
                const pageData = await this.renderPageFixed(pageNum, { scale: newScale });
                
                // Update cache
                this.renderedPages.set(pageNum, pageData);
                this.updateCacheOrder(pageNum);
                
                console.log(`Re-rendered visible page ${pageNum} at scale ${newScale}`);
        } catch (error) {
                console.warn(`Error re-rendering visible page ${pageNum}:`, error);
            }
        }
    }

    /**
     * Re-render current pages at new zoom level
     */
    async rerenderCurrentPages(zoomLevel) {
        const currentPages = Array.from(this.renderedPages.keys());
        
        for (const pageNum of currentPages) {
            try {
                // Get new scale with zoom level
                const newScale = this.getAdaptiveScale(false, zoomLevel);
                
                // Re-render page at new scale
                const pageData = await this.renderPageAtScale(pageNum, newScale);
                
                // Update cache
                this.renderedPages.set(pageNum, pageData);
                this.updateCacheOrder(pageNum);
                
                console.log(`Re-rendered page ${pageNum} at zoom level ${zoomLevel}`);
            } catch (error) {
                console.warn(`Error re-rendering page ${pageNum}:`, error);
            }
        }
    }

    /**
     * Create multi-scale tiles for different zoom levels
     */
    async createMultiScaleTiles(pageNumber) {
        const scales = [0.5, 1.0, 1.5, 2.0]; // Different zoom levels
        const tiles = new Map();
        
        for (const scale of scales) {
            try {
                const tileData = await this.renderPageAtScale(pageNumber, scale);
                tiles.set(scale, tileData);
                console.log(`Created tile for page ${pageNumber} at scale ${scale}`);
            } catch (error) {
                console.warn(`Error creating tile for page ${pageNumber} at scale ${scale}:`, error);
            }
        }
        
        return tiles;
    }

    /**
     * Get appropriate tile based on current zoom level
     */
    getTileForZoomLevel(pageNumber, zoomLevel) {
        const scales = [0.5, 1.0, 1.5, 2.0];
        const targetScale = Math.min(zoomLevel, 2.0);
        
        // Find closest scale
        const closestScale = scales.reduce((prev, curr) => 
            Math.abs(curr - targetScale) < Math.abs(prev - targetScale) ? curr : prev
        );
        
        const cacheKey = `${pageNumber}_${closestScale}`;
        return this.multiResolutionCache.get(cacheKey);
    }
}

// Global instance
window.PDFRenderer = PDFRenderer;

