/**
 * Vertical Side Menu Drawbar JavaScript
 * Handles hamburger menu and drawer interactions
 */

class VerticalSideMenu {
    constructor() {
        this.menu = document.getElementById('vertical-side-menu');
        this.hamburgerMenu = document.getElementById('hamburger-menu');
        this.sidebarDrawer = document.getElementById('sidebar-drawer');
        this.drawerClose = document.getElementById('drawer-close');
        this.drawerMenuItems = document.querySelector('.drawer-menu-items');
        this.isOpen = false;
        
        this.init();
    }
    
    init() {
        if (!this.menu || !this.hamburgerMenu || !this.sidebarDrawer) {
            return;
        }
        
        this.createOverlay();
        this.bindEvents();
        this.setupMenuItems();
        this.addAnimationClass();
    }
    
    createOverlay() {
        this.overlay = document.createElement('div');
        this.overlay.className = 'drawer-overlay';
        this.overlay.id = 'drawer-overlay';
        document.body.appendChild(this.overlay);
    }
    
    bindEvents() {
        // Toggle drawer on hamburger menu click
        this.hamburgerMenu.addEventListener('click', (e) => {
            e.preventDefault();
            this.toggleDrawer();
        });
        
        // Close drawer on close button click
        this.drawerClose.addEventListener('click', (e) => {
            e.preventDefault();
            this.closeDrawer();
        });
        
        // Close drawer when clicking overlay
        this.overlay.addEventListener('click', () => {
            this.closeDrawer();
        });
        
        // Handle escape key
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && this.isOpen) {
                this.closeDrawer();
            }
        });
        
        // Handle window resize
        window.addEventListener('resize', () => {
            this.handleResize();
        });
        
        // Handle scroll to update active menu item
        window.addEventListener('scroll', () => {
            this.updateActiveOnScroll();
        });
    }
    
    setupMenuItems() {
        const menuLinks = this.drawerMenuItems.querySelectorAll('.drawer-menu-link');
        
        menuLinks.forEach((link, index) => {
            link.addEventListener('click', (e) => {
                // Only prevent default for internal navigation (first 3 items)
                if (index < 3) {
                    e.preventDefault();
                }
                this.handleMenuItemClick(link, index);
            });
        });
    }
    
    toggleDrawer() {
        if (this.isOpen) {
            this.closeDrawer();
        } else {
            this.openDrawer();
        }
    }
    
    openDrawer() {
        this.isOpen = true;
        this.sidebarDrawer.classList.add('open');
        this.overlay.classList.add('active');
        document.body.style.overflow = 'hidden';
        
        // Add staggered animation to menu items
        const items = this.drawerMenuItems.querySelectorAll('.drawer-menu-item');
        items.forEach((item, index) => {
            item.style.animationDelay = `${index * 0.1}s`;
            item.classList.add('animate-in');
        });
        
        // Store state in localStorage
        localStorage.setItem('verticalMenuOpen', 'true');
    }
    
    closeDrawer() {
        this.isOpen = false;
        this.sidebarDrawer.classList.remove('open');
        this.overlay.classList.remove('active');
        document.body.style.overflow = '';
        
        // Remove animation classes
        const items = this.drawerMenuItems.querySelectorAll('.drawer-menu-item');
        items.forEach(item => {
            item.classList.remove('animate-in');
        });
        
        // Store state in localStorage
        localStorage.setItem('verticalMenuOpen', 'false');
    }
    
    handleMenuItemClick(link, index) {
        const icon = link.querySelector('i');
        const tooltip = link.closest('.drawer-menu-item').getAttribute('data-tooltip');
        const href = link.getAttribute('href');
        
        // Update active class
        this.updateActiveMenuItem(index);
        
        // Add click animation
        link.closest('.drawer-menu-item').classList.add('clicked');
        setTimeout(() => {
            link.closest('.drawer-menu-item').classList.remove('clicked');
        }, 200);
        
        // Close drawer after action
        setTimeout(() => {
            this.closeDrawer();
        }, 300);
        
        // Handle different menu actions based on index
        switch(index) {
            case 0: // Home
                this.scrollToSection('#home');
                break;
            case 1: // Finance at a glance
                this.scrollToSection('#finance-at-a-glance');
                break;
            case 2: // Remarks
                this.scrollToSection('#remarks');
                break;
            case 3: // Welcome - Main website
                window.location.href = href;
                break;
            default:
        }
    }
    
    scrollToSection(selector) {
        const element = document.querySelector(selector);
        if (element) {
            element.scrollIntoView({
                behavior: 'smooth',
                block: 'start'
            });
        } else {
            const allElements = document.querySelectorAll('[id*="' + selector.replace('#', '') + '"]');
            if (allElements.length > 0) {
            }
        }
    }
    
    updateActiveMenuItem(activeIndex) {
        const menuItems = this.drawerMenuItems.querySelectorAll('.drawer-menu-item');
        menuItems.forEach(item => {
            item.classList.remove('active');
        });
        
        if (menuItems[activeIndex]) {
            menuItems[activeIndex].classList.add('active');
        }
    }
    
    updateActiveOnScroll() {
        const sections = ['#home', '#finance-at-a-glance', '#remarks'];
        const menuItems = this.drawerMenuItems.querySelectorAll('.drawer-menu-item');
        
        let currentSection = '';
        
        sections.forEach(section => {
            const element = document.querySelector(section);
            if (element) {
                const rect = element.getBoundingClientRect();
                if (rect.top <= 100 && rect.bottom >= 100) {
                    currentSection = section;
                }
            }
        });
        
        if (currentSection) {
            const sectionIndex = sections.indexOf(currentSection);
            if (sectionIndex !== -1) {
                this.updateActiveMenuItem(sectionIndex);
            }
        }
    }
    
    handleEditAction() {
        this.showNotification('Edit mode activated');
    }
    
    handleReportsAction() {
       
        window.open('/reports', '_blank');
    }

    handleShareAction() {
        // Implement share functionality
        if (navigator.share) {
            navigator.share({
                title: document.title,
                url: window.location.href
            });
        } else {
            // Fallback: copy to clipboard
            navigator.clipboard.writeText(window.location.href).then(() => {
                this.showNotification('Link copied to clipboard');
            });
        }
    }
    
    handleLocationAction() {
        this.showNotification('Location services activated');
    }
    
    
    showNotification(message) {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = 'vertical-menu-notification';
        notification.textContent = message;
        
        // Style the notification
        Object.assign(notification.style, {
            position: 'fixed',
            top: '20px',
            right: '20px',
            background: '#1e3a8a',
            color: 'white',
            padding: '12px 20px',
            borderRadius: '8px',
            fontSize: '14px',
            fontWeight: '500',
            zIndex: '10000',
            opacity: '0',
            transform: 'translateX(100%)',
            transition: 'all 0.3s ease'
        });
        
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateX(0)';
        }, 100);
        
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                document.body.removeChild(notification);
            }, 300);
        }, 3000);
    }
    
    handleResize() {
        if (window.innerWidth < 768 && this.isOpen) {
            this.closeDrawer();
        }
    }
    
    addAnimationClass() {
        setTimeout(() => {
            this.menu.classList.add('animate-in');
        }, 100);
    }
    
    open() {
        this.openDrawer();
    }
    
    close() {
        this.closeDrawer();
    }
    
    isMenuOpen() {
        return this.isOpen;
    }
}

// Initialize the vertical menu when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    
    const wasOpen = localStorage.getItem('verticalMenuOpen') === 'true';
    
    window.verticalMenu = new VerticalSideMenu();
    
    if (wasOpen) {
        setTimeout(() => {
            window.verticalMenu.open();
        }, 500);
    }
});

setTimeout(function() {
    if (!window.verticalMenu) {
        window.verticalMenu = new VerticalSideMenu();
    }
}, 1000);

if (typeof module !== 'undefined' && module.exports) {
    module.exports = VerticalSideMenu;
}
